--------------------------------------------------------------------------
-- Gymnast.lua 
--------------------------------------------------------------------------
--[[
Gymnast Tooltips

Bending over backwards to please. Make your Tooltip as flexible as you are.

By author: AnduinLothar    <karlkfi@cosmosui.org>

Slash Commands:
"/gtshow" - Makes the Game Tooltip visible for you to drag to your preferred position.
"/gtreset" - Moves the Game Tooltip back to its original position.
"/gttopcenter" - Moves the Game Tooltip to the alternate top center position.
"/gtanchor [position]" - Makes the tooltip extend from this anchored position.
(position can be "SMART", "TOPLEFT", "TOPRIGHT", "BOTTOMLEFT", "BOTTOMRIGHT", "LEFT", "TOP", "RIGHT", "BOTTOM", "MOUSE", "UBER", "OWNER")
Note: "MOUSE", "UBER", and "OWNER" are toggle commands. Call them again to disable.

'mouse' relocates the tooltip to the mouse.
'uber' relocates "Enhanced Tooltips" to show near their parent frame. (toggle)
'owner' relocates normal tooltips to show near their parent frame. (toggle)

	$Id: Gymnast.lua 4174 2006-10-14 10:25:03Z karlkfi $
	$Rev: 4174 $
	$LastChangedBy: karlkfi $
	$Date: 2006-10-14 05:25:03 -0500 (Sat, 14 Oct 2006) $
	
changelog:
v3.0
-Updated for lua 5.1
-Updated TOC to 20000
-Anchor to MOUSE mode now correctly dissables if you set another mode.
-OO'd the code
-Removed Chronos Dep
v2.7
-Added an option to relocate normal tooltips to be by their parent, like Uber Tooltips were.  Using this option will fix things such as Buff tooltips showing up off screen when you move the buffs to the left of the screen.
-Fixed bug with the position by owner option, needed GetEffectiveScale
-Updated to not use RegisterForSave
-Updated TOC to 11000
v2.6
-Updated TOC to 1700
-Fixed issues not accounting for Screen Scaling
-Fixed compatibility with CosmosLegacy.
v2.5
-Updated TOC to 1600
-Added Khaos Options.
-Fixed the Smart Anchor option to be under advanced options.
-Raised advanced options to Master difficulty.
-Made feedback localizable.
-Clarified Uber tooltip relocation in the mouse-over info.
v2.42
-Extra options now save correctly when not using cosmos.
v2.41
-Updated TOC to 1300
-Fixed a Cosmos dependancy to be truely optional
v2.4
-French localization updated by Sasmira
v2.3
-Added smart anchoring and center anchoring
-Added a button to move to the old cosmos position top center.
v2.2
-Added Mouse and Uber tooltip relocations
v2.1
-Added Anchoring
-Modified instructions in the tooltip.
v2.0
-Completely redesigned... now it plays nicely with TooltipsBase.
v1.0
-Replaced tooltips with mobile ones.
-Included instructions in the tooltip.

]]--

Gymnast = {};

-- Constants
Gymnast.TOOLTIPS_POINT = "Point";
Gymnast.TOOLTIPS_RELATIVETO = "RelativeTo";
Gymnast.TOOLTIPS_RELATIVEPOINT = "RelativePoint";
Gymnast.TOOLTIPS_POSITION = "Position";

local SavedGameTooltip_SetDefaultAnchor = nil;
local SavedSmartSetOwner = nil;
local SavedGameTooltip_SetOwner = nil;
local anchorPositions = { "TOPLEFT", "TOPRIGHT", "BOTTOMLEFT", "BOTTOMRIGHT", "CENTER", "LEFT", "TOP", "RIGHT", "BOTTOM" };
Gymnast.SetTooltipAnchorList = {};


function Gymnast.OnLoad()
	
	this:RegisterEvent("VARIABLES_LOADED");
	
	if (GameTooltip_SetDefaultAnchor ~= SavedGameTooltip_SetDefaultAnchor) then
		SavedGameTooltip_SetDefaultAnchor = GameTooltip_SetDefaultAnchor;
		GameTooltip_SetDefaultAnchor = Gymnast.GameTooltip_SetDefaultAnchor;
	end

	if (SmartSetOwner ~= SavedSmartSetOwner) then
		SavedSmartSetOwner = SmartSetOwner;
		SmartSetOwner = Gymnast.SmartSetOwner;
	end

	if (GameTooltip.SetOwner ~= SavedGameTooltip_SetOwner) then
		SavedGameTooltip_SetOwner = GameTooltip.SetOwner;
		GameTooltip.SetOwner = Gymnast.GameTooltip_SetOwner;
	end
	
	for i, anchor in ipairs(anchorPositions) do
		Gymnast.SetTooltipAnchorList[anchor] = function() Gymnast.SetTooltipAnchor(anchor) end;
	end
	
	if ( Khaos ) then 
		local options = {};
		local commands = {};
		local optionSet = {
			id="Gymnast";
			text=GYMNAST_TOOLTIPS_HEADER;
			helptext=GYMNAST_TOOLTIPS_HEADER_INFO;
			difficulty=1;
			options=options;
			commands=commands;
		};

		table.insert(
			options, 
			{
				id="Header";
				type=K_HEADER;
				text=TEXT(GYMNAST_TOOLTIPS_HEADER);
				helptext=TEXT(GYMNAST_TOOLTIPS_HEADER_INFO);
				difficulty=1;
				default=false;
			}
		);
		table.insert(
			options, 
			{
				id="Enable";
				type=K_TEXT;
				text=TEXT(GYMNAST_TOOLTIPS_ENABLED_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_ENABLED_TEXT_INFO);
				callback=function(state) 
					Gymnast.Enabled=state.checked;
				end;
				feedback=function(state)
					if ( state.checked ) then 
						return GYMNAST_TOOLTIPS_ENABLED_TEXT_FEEDBACK_ON;
					else
						return GYMNAST_TOOLTIPS_ENABLED_TEXT_FEEDBACK_OFF;
					end
				end;
				difficulty=1;
				check=true;
				default={checked=true};
				disabled={checked=false};
			}
		);
		table.insert(
			options, 
			{
				id="OwnerAnchor";
				type=K_TEXT;
				text=TEXT(GYMNAST_TOOLTIPS_OWNER_ANCHOR_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_OWNER_ANCHOR_TEXT_INFO);
				callback=function(state) 
					Gymnast.OwnerAnchor=state.checked;
				end;										
				feedback=function(state)
					if ( state.checked ) then 
						return GYMNAST_TOOLTIPS_OWNER_ANCHOR_TEXT_FEEDBACK_ON;
					else
						return GYMNAST_TOOLTIPS_OWNER_ANCHOR_TEXT_FEEDBACK_OFF;
					end
				end;
				difficulty=1;
				check=true;
				default={checked=false};
				disabled={checked=false};
			}
		);
		table.insert(
			options, 
			{
				id="UberAnchor";
				type=K_TEXT;
				text=TEXT(GYMNAST_TOOLTIPS_ANCHOR_UBER_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_ANCHOR_UBER_TEXT_INFO);
				callback=function(state) 
					Gymnast_RelocateUberTooltips=state.checked;
				end;										
				feedback=function(state)
					if ( state.checked ) then 
						return GYMNAST_TOOLTIPS_ANCHOR_UBER_TEXT_FEEDBACK_ON;
					else
						return GYMNAST_TOOLTIPS_ANCHOR_UBER_TEXT_FEEDBACK_OFF;
					end
				end;
				difficulty=1;
				check=true;
				default={checked=false};
				disabled={checked=false};
			}
		);
		table.insert(
			options, 
			{
				id="AnchorMouse";
				type=K_TEXT;
				text=TEXT(GYMNAST_TOOLTIPS_ANCHOR_MOUSE_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_ANCHOR_MOUSE_TEXT_INFO);
				callback=function(state) 
					Gymnast_RelocateTooltip_ToMouse=state.checked;
				end;										
				feedback=function(state)
					if ( state.checked ) then 
						return GYMNAST_TOOLTIPS_ANCHOR_MOUSE_TEXT_FEEDBACK_ON;
					else
						return GYMNAST_TOOLTIPS_ANCHOR_MOUSE_TEXT_FEEDBACK_OFF;
					end
				end;
				difficulty=1;
				check=true;
				default={checked=false};
				disabled={checked=false};
			}
		);
		table.insert(
			options, 
			{
				id="ShowMobile";
				type=K_BUTTON;
				text=TEXT(GYMNAST_TOOLTIPS_SHOW_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_SHOW_TEXT_INFO);
				callback=Gymnast.ShowMobileGameTooltip;										
				difficulty=1;
				setup={buttonText=SHOW};
			}
		);
		table.insert(
			options, 
			{
				id="ResetTooltip";
				type=K_TEXT;
				text=TEXT(GYMNAST_TOOLTIPS_RESET_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_RESET_TEXT_INFO);
				callback=Gymnast.ResetGameTooltip;									
				difficulty=1;
				setup={buttonText=RESET};
			}
		);
		table.insert(
			options, 
			{
				id="TopCenter";
				type=K_BUTTON;
				text=TEXT(GYMNAST_TOOLTIPS_SHOW_TOPCENTER_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_SHOW_TOPCENTER_TEXT_INFO);
				callback=Gymnast.SetTooltipTopCenter;									
				difficulty=1;
				setup={buttonText=SHOW};
			}
		);
		table.insert(
			options, 
			{
				id="AdvancedOptionsHeader";
				type=K_HEADER;
				text=TEXT(GYMNAST_TOOLTIPS_ADV_OPTIONS);
				helptext=TEXT(GYMNAST_TOOLTIPS_ADV_OPTIONS_INFO);
				difficulty=3;
			}
		);
		table.insert(
			options, 
			{
				id="SmartAnchor";
				type=K_TEXT;
				text=TEXT(GYMNAST_TOOLTIPS_ANCHOR_SMART_TEXT);
				helptext=TEXT(GYMNAST_TOOLTIPS_ANCHOR_SMART_TEXT_INFO);
				callback=function(state) 
					Gymnast_TooltipAnchorSmart=state.checked;
				end;										
				feedback=function(state)
					if ( state.checked ) then 
						return GYMNAST_TOOLTIPS_ANCHOR_SMART_TEXT_FEEDBACK_ON;
					else
						return GYMNAST_TOOLTIPS_ANCHOR_SMART_TEXT_FEEDBACK_OFF;
					end
				end;
				difficulty=3;
				check=true;
				default={checked=true};
				disabled={checked=true};
			}
		);

	for index, position in ipairs(anchorPositions) do
		table.insert(
			options, 
			{
				id="Anchor"..Sea.string.capitalizeWords(position);
				type=K_BUTTON;
				text=string.format(GYMNAST_TOOLTIPS_ANCHOR_TEXT,position);
				helptext=string.format(GYMNAST_TOOLTIPS_ANCHOR_TEXT_INFO,position);
				callback=Gymnast.SetTooltipAnchorList[position];										
				difficulty=3;
				setup={buttonText=position};
			}
		);
	end
	table.insert(commands, 
		{
			id="GymnastCommands";
			commands={"/gt"};
			parseTree={
				["show"] = Gymnast.ShowMobileGameTooltip;
				["topcenter"] = Gymnast.SetTooltipTopCenter;
				["reset"] = Gymnast.ResetGameTooltip;
				["anchor"] = Gymnast.SlashAnchorTooltip;
			};
		}
	);
		Khaos.registerOptionSet(
			"tooltip",
			optionSet
		);
	elseif (Cosmos_RegisterConfiguration) then
		--register Mobile Frames with cosmos config
		Cosmos_RegisterConfiguration("COS_GYMNAST_TOOLTIPS_HEADER",
			"SECTION",
			TEXT(GYMNAST_TOOLTIPS_HEADER),
			TEXT(GYMNAST_TOOLTIPS_HEADER_INFO)
		);
		Cosmos_RegisterConfiguration("COS_GYMNAST_TOOLTIPS_HEADER_SECTION",
			"SEPARATOR",
			TEXT(GYMNAST_TOOLTIPS_HEADER),
			TEXT(GYMNAST_TOOLTIPS_HEADER_INFO)
		);
		
		Cosmos_RegisterConfiguration(
			"COS_GYMNAST_TOOLTIPS_ENABLED",
			"CHECKBOX",
			TEXT(GYMNAST_TOOLTIPS_ENABLED_TEXT),
			TEXT(GYMNAST_TOOLTIPS_ENABLED_TEXT_INFO),
			Gymnast.ToggleEnable,
			0
		);
		
		Cosmos_RegisterConfiguration(
			"COS_GYMNAST_TOOLTIPS_ANCHOR_UBER",
			"CHECKBOX",
			TEXT(GYMNAST_TOOLTIPS_ANCHOR_UBER_TEXT),
			TEXT(GYMNAST_TOOLTIPS_ANCHOR_UBER_TEXT_INFO),
			Gymnast.GameTooltipUber,
			0
		);
		
		Cosmos_RegisterConfiguration(
			"COS_GYMNAST_TOOLTIPS_ANCHOR_MOUSE",
			"CHECKBOX",
			TEXT(GYMNAST_TOOLTIPS_ANCHOR_MOUSE_TEXT),
			TEXT(GYMNAST_TOOLTIPS_ANCHOR_MOUSE_TEXT_INFO),
			Gymnast.RelocateGameTooltipToMouse,
			0
		);
		
		--show button
		Cosmos_RegisterConfiguration("COS_GYMNAST_TOOLTIPS_SHOW",	-- registered name
			"BUTTON",												-- register type
			TEXT(GYMNAST_TOOLTIPS_SHOW_TEXT),								-- short description
			TEXT(GYMNAST_TOOLTIPS_SHOW_TEXT_INFO),						-- long description
			Gymnast.ShowMobileGameTooltip,								-- function
			0,						-- useless
			0,						-- useless
			0,						-- useless
			0,						-- useless
			TEXT(SHOW)		-- button text
		);
		--reset button
		Cosmos_RegisterConfiguration("COS_GYMNAST_TOOLTIPS_RESET",  -- registered name
			"BUTTON",												-- register type
			TEXT(GYMNAST_TOOLTIPS_RESET_TEXT),							-- short description
			TEXT(GYMNAST_TOOLTIPS_RESET_TEXT_INFO),						-- long description
			Gymnast.ResetGameTooltip,								-- function
			0,						-- useless
			0,						-- useless
			0,						-- useless
			0,						-- useless
			TEXT(RESET)		-- button text
		);
		
		Cosmos_RegisterConfiguration("COS_GYMNAST_TOOLTIPS_SHOW_TOPCENTER",  -- registered name
			"BUTTON",												-- register type
			TEXT(GYMNAST_TOOLTIPS_SHOW_TOPCENTER_TEXT),							-- short description
			TEXT(GYMNAST_TOOLTIPS_SHOW_TOPCENTER_TEXT_INFO),						-- long description
			Gymnast.SetTooltipTopCenter,								-- function
			0,						-- useless
			0,						-- useless
			0,						-- useless
			0,						-- useless
			TEXT(SHOW)		-- button text
		);
		
		Cosmos_RegisterConfiguration(
			"COS_GYMNAST_TOOLTIPS_ANCHOR_SMART",  -- registered name
			"CHECKBOX",													--Things to use
			TEXT(GYMNAST_TOOLTIPS_ANCHOR_SMART_TEXT),
			TEXT(GYMNAST_TOOLTIPS_ANCHOR_SMART_TEXT_INFO),
			Gymnast.SetTooltipAnchorSmart,								--Callback
			1										--Default Checked/Unchecked
		);
		
		Cosmos_RegisterConfiguration("COS_GYMNAST_TOOLTIPS_ADV_OPTIONS_SECTION",
			"SEPARATOR",
			TEXT(GYMNAST_TOOLTIPS_ADV_OPTIONS),
			TEXT(GYMNAST_TOOLTIPS_ADV_OPTIONS_INFO)
		);
		
		for index, position in ipairs(anchorPositions) do
			Cosmos_RegisterConfiguration(
				"COS_GYMNAST_TOOLTIPS_ANCHOR_"..position,				-- registered name
				"BUTTON",												-- register type
				format(GYMNAST_TOOLTIPS_ANCHOR_TEXT, position),			-- short description
				format(GYMNAST_TOOLTIPS_ANCHOR_TEXT_INFO, position),	-- long description
				Gymnast.SetTooltipAnchorList[position],					-- function
				0,						-- useless
				0,						-- useless
				0,						-- useless
				0,						-- useless
				TEXT(position)		-- button text
			);
		end
		
	else
		Gymnast.Enabled = true;
--		RegisterForSave("Gymnast_RelocateUberTooltips");
--		RegisterForSave("Gymnast_RelocateTooltip_ToMouse");
--		RegisterForSave("Gymnast_TooltipAnchorSmart");
	end
	
	SlashCmdList["GYMNASTTOOLTIPSSHOW"] = Gymnast.ShowMobileGameTooltip;
    SLASH_GYMNASTTOOLTIPSSHOW1 = "/gtshow";
	
	SlashCmdList["GYMNASTTOOLTIPSTOPCENTER"] = Gymnast.SetTooltipTopCenter;
    SLASH_GYMNASTTOOLTIPSTOPCENTER1 = "/gttopcenter";
	
	SlashCmdList["GYMNASTTOOLTIPSRESET"] = Gymnast.ResetGameTooltip;
    SLASH_GYMNASTTOOLTIPSRESET1 = "/gtreset";
	
	SlashCmdList["GYMNASTTOOLTIPSANCHOR"] = Gymnast.SlashAnchorTooltip;
    SLASH_GYMNASTTOOLTIPSANCHOR1 = "/gtanchor";
	
end


function Gymnast.OnEvent()
	if (event == "VARIABLES_LOADED") then
		if (not Gymnast_TooltipPointSettings) then
		Gymnast_TooltipPointSettings = {
			[Gymnast.TOOLTIPS_POINT] = "TOPLEFT",
			[Gymnast.TOOLTIPS_RELATIVETO] = "UIParent",
			[Gymnast.TOOLTIPS_RELATIVEPOINT] = "BOTTOMLEFT",
			[Gymnast.TOOLTIPS_POSITION] = nil
		};
		end
	end
end


function Gymnast.SetTooltipAnchor(point)
	Gymnast.SetTooltipAnchorSmart(0);
	if (CosmosMasterFrame) and (Cosmos_RegisterConfiguration) then
		Cosmos_UpdateValue("COS_GYMNAST_TOOLTIPS_ANCHOR_SMART", CSM_CHECKONOFF, 0);
		if CosmosMasterFrame:IsVisible() and (not CosmosMasterFrame_IsLoading) then
			CosmosMaster_DrawData();
		end
	end
	Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = point;
end

function Gymnast.SetTooltipTopCenter()
	MobileGameTooltip:SetOwner(UIParent, "ANCHOR_NONE");
	MobileGameTooltip:SetPoint("TOP", "UIParent", "TOP", 0, -32);
	local r = 0.3;
	local g = 0.7;
	local b = 0.9;
	MobileGameTooltip:AddLine(GYMNAST_TOOLTIPS_HEADER, r+.3, g, b);
	MobileGameTooltip:AddLine(" ", r, g, b);
	MobileGameTooltip:AddLine(GYMNAST_TOOLTIPS_INSTRUCTIONS, r, g, b);
	MobileGameTooltip:Show();
end

function Gymnast.SetTooltipAnchorSmart(checked)
	Gymnast_TooltipAnchorSmart = (checked == 1);
end

function Gymnast.SlashAnchorTooltip(msg)
	local var;
	local onVal;
	msg = string.upper(msg);
	local startpos, endpos, anchor = string.find(msg, "(%w+)");
	if (anchor) then
		Gymnast_RelocateTooltip_ToMouse = false;
		if (anchor == "SMART") then
			Gymnast.SetTooltipAnchorSmart(1);
			if (CosmosMasterFrame) and (Cosmos_RegisterConfiguration) then
				Cosmos_UpdateValue("COS_GYMNAST_TOOLTIPS_ANCHOR_SMART", CSM_CHECKONOFF, 1);
				if CosmosMasterFrame:IsVisible() and (not CosmosMasterFrame_IsLoading) then
					CosmosMaster_DrawData();
				end
			end
		elseif (anchor == "MOUSE") then
			Gymnast.RelocateGameTooltipToMouse();
		elseif (anchor == "OWNER") then
			Gymnast.GameTooltipOwner();
		elseif (anchor == "UBER") then
			Gymnast.GameTooltipUber();
		else
			for index, position in ipairs(anchorPositions) do
				if (position == anchor) then
					Gymnast.SetTooltipAnchor(position);
				end
			end
		end
	end
end


function Gymnast.MarkTooltipCoords()
	local centerX,centerY = MobileGameTooltip:GetCenter();
	local left = MobileGameTooltip:GetLeft(); 
	local top = MobileGameTooltip:GetTop(); 
	local right = MobileGameTooltip:GetRight(); 
	local bottom = MobileGameTooltip:GetBottom();
	local coords;
	local place = Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT];
	
	if (Gymnast_TooltipAnchorSmart) then
		local parentW = UIParent:GetRight() - UIParent:GetLeft();
		local parentH = UIParent:GetTop() - UIParent:GetLeft();
		local onRightThird = (centerX > 2*(parentW/3));
		local onLeftThird = (centerX < (parentW/3));
		local onTopThird = (centerY > 2*(parentH/3));
		local onBottomThird = (centerY < (parentH/3));
		if onRightThird then
			if onTopThird then
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "TOPRIGHT";
			elseif onBottomThird then
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "BOTTOMRIGHT";
			else
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "RIGHT";
			end
		elseif onLeftThird then
			if onTopThird then
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "TOPLEFT";
			elseif onBottomThird then
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "BOTTOMLEFT";
			else
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "LEFT"; 
			end
		else
			if onTopThird then
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "TOP"; 
			elseif onBottomThird then
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "BOTTOM";
			else
				Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT] = "CENTER";
			end
		end
	end
	
	local place = Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT];
	
	if (place == "TOPLEFT") then
		coords = { left, top }; 
	elseif (place == "TOPRIGHT") then
		coords = { right, top }; 
	elseif (place == "BOTTOMLEFT") then
		coords = { left, bottom }; 
	elseif (place == "BOTTOMRIGHT") then
		coords = { right, bottom };
	elseif (place == "CENTER") then
		coords = { centerX, centerY }; 
	elseif (place == "LEFT") then
		coords = { left, centerY }; 
	elseif (place == "TOP") then
		coords = { centerX, top }; 
	elseif (place == "RIGHT") then
		coords = { right, centerY };
	elseif (place == "BOTTOM") then
		coords = { centerX, bottom };
	end
	Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POSITION] = coords;
end

function Gymnast.RelocateGameTooltipToMouse(toggle)
	if ( toggle == nil ) then
		Gymnast_RelocateTooltip_ToMouse = (not Gymnast_RelocateTooltip_ToMouse);
	elseif ( toggle == 0 ) then
		Gymnast_RelocateTooltip_ToMouse = false;
	else
		Gymnast_RelocateTooltip_ToMouse = true;
	end
end

function Gymnast.GameTooltipUber(toggle)
	if ( toggle == nil ) then
		Gymnast_RelocateUberTooltips = (not Gymnast_RelocateUberTooltips);
	elseif ( toggle == 0 ) then
		Gymnast_RelocateUberTooltips = false;
	else
		Gymnast_RelocateUberTooltips = true;
	end
end

function Gymnast.GameTooltipOwner(toggle)
	if ( toggle == nil ) then
		Gymnast.OwnerAnchor = (not Gymnast.OwnerAnchor);
	elseif ( toggle == 0 ) then
		Gymnast.OwnerAnchor = false;
	else
		Gymnast.OwnerAnchor = true;
	end
end

function Gymnast.ToggleEnable(toggle)
	if ( toggle == 0 ) then
		Gymnast.Enabled = false;
	else
		Gymnast.Enabled = true;
	end
end

function Gymnast.GameTooltip_SetOwner(frame, owner, anchorPoint, setX, setY)
	if (Gymnast.Enabled and Gymnast.OwnerAnchor) then
		Gymnast.SmartSetOwner(owner, frame, setX, setY);
	else
		SavedGameTooltip_SetOwner(frame, owner, anchorPoint, setX, setY);
	end
end

function Gymnast.GameTooltip_SetDefaultAnchor(tooltip, owner) 
	if (Gymnast.Enabled) then
		if (Gymnast_RelocateUberTooltips) and (owner ~= UIParent) then
			NewSmartSetOwner(owner, "ANCHOR_NONE", 0, 0, tooltip);
			return;
		elseif (tooltip == GameTooltip) or (tooltip == MobileGameTooltip) then 
			if (Gymnast_RelocateTooltip_ToMouse) then
				tooltip:ClearAllPoints();
				if (tooltip == GameTooltip) then
					SavedGameTooltip_SetOwner(tooltip, UIParent, "ANCHOR_CURSOR");
				else
					tooltip:SetOwner(UIParent, "ANCHOR_CURSOR");
				end
				return;
			elseif ( Gymnast_TooltipPointSettings ) then
				local position = Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POSITION];
				if ( ( position ) and ( position[1] ) and ( position[2] ) ) then
					tooltip:Hide()
					if (tooltip == GameTooltip) then
						SavedGameTooltip_SetOwner(tooltip, owner, "ANCHOR_NONE");
					else
						tooltip:SetOwner(owner, "ANCHOR_NONE");
					end
					tooltip:ClearAllPoints();
					tooltip:SetPoint(Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POINT], Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_RELATIVETO], Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_RELATIVEPOINT], position[1], position[2] );
					return;
				end
			end
		end
	end
	SavedGameTooltip_SetDefaultAnchor(tooltip, owner);
end

function Gymnast.SmartSetOwner(owner, tooltip, setX, setY)
	if (not owner) then
 		owner = this;
 	end
 	if (not tooltip) then
 		tooltip = GameTooltip;
 	end
	NewSmartSetOwner(owner, "ANCHOR_NONE", setX, setY, tooltip);
end

function NewSmartSetOwner(owner, position, setX, setY, tooltip)
	if (not owner) then
 		owner = UIParent;
 	end
 	if (not tooltip) then
 		tooltip = this;
 	end
	if (not setX) then
 		setX = 0;
 	end
 	if (not setY) then
 		setY = 0;
 	end

	local x,y = owner:GetCenter();
	local left = owner:GetLeft();
	local right = owner:GetRight();
	local top = owner:GetTop();
	local bottom = owner:GetBottom();
	local screenWidth = UIParent:GetRight() - UIParent:GetLeft();
	local screenHeight = UIParent:GetTop() - UIParent:GetBottom();
	local scale = owner:GetEffectiveScale();
	if (x~=nil and y~=nil and left~=nil and right~=nil and top~=nil and bottom~=nil and screenWidth>0 and screenHeight>0) then
		setX = setX * scale;
		setY = setY * scale;
		x = x * scale;
		y = y * scale;
		left = left * scale;
		right = right * scale;
		width = right - left;
		top = top * scale;
		bottom = bottom * scale;
		local anchorPoint = "";
		if (y <= (screenHeight * (1/2))) then
			top = top + setY;
			anchorPoint = "TOP";
			if (top < 0) then
				setY = setY - top;
			end
		else
			setY = -setY;
			bottom = bottom + setY;
			anchorPoint = "BOTTOM";
			if (bottom > screenHeight) then
				setY = setY + (screenHeight - bottom);
			end
		end
		
		if (x <= (screenWidth * (1/2))) then
			left = left + setX;
			if (anchorPoint == "BOTTOM") then
				anchorPoint = anchorPoint.."RIGHT";
				setX = setX - width;
				if (left < 0) then
					setX = setX - left;
				end
			else
				anchorPoint = anchorPoint.."LEFT";
				if (left < 0) then
					setX = setX - left;
				end
			end
		else
			setX = -setX;
			right = right + setX;
			if (anchorPoint == "BOTTOM") then
				anchorPoint = anchorPoint.."LEFT";
				setX = setX + width;
				if (right > screenWidth) then
					setX = setX - (right - screenWidth);
				end
			else
				anchorPoint = anchorPoint.."RIGHT";
				if (right > screenWidth) then
					setX = setX + (screenWidth - right);
				end
			end
		end
		
		if (anchorPoint == "") then
			anchorPoint = "TOPLEFT";
		end
		scale = tooltip:GetScale();
		if (scale) then
			setX = setX / scale;
			setY = setY / scale;
		end
		if (tooltip == GameTooltip) then
			SavedGameTooltip_SetOwner(tooltip, owner, "ANCHOR_"..anchorPoint, setX, setY);
		else
			tooltip:SetOwner(owner, "ANCHOR_"..anchorPoint, setX, setY);
		end
	end
end

function Gymnast.ShowMobileGameTooltip()
	GameTooltip_SetDefaultAnchor(MobileGameTooltip, UIParent);
	local r = 0.3;
	local g = 0.7;
	local b = 0.9;
	MobileGameTooltip:AddLine(GYMNAST_TOOLTIPS_HEADER, r+.3, g, b);
	MobileGameTooltip:AddLine(" ", r, g, b);
	MobileGameTooltip:AddLine(GYMNAST_TOOLTIPS_INSTRUCTIONS, r, g, b);
	MobileGameTooltip:Show();
end

function Gymnast.ResetGameTooltip()
	Gymnast_TooltipPointSettings[Gymnast.TOOLTIPS_POSITION] = nil;
	Gymnast.ShowMobileGameTooltip();
	MobileGameTooltip.fadeCount = 1;
end

-- Frame Scripts
function Gymnast.OnMouseUp()
	if ( this.isMoving ) then
		this:StopMovingOrSizing();
		this.isMoving = false;
	end
	Gymnast.MarkTooltipCoords();
	this.fadeCount = 1;
end

function Gymnast.OnMouseDown()
	if ( ( not this.isLocked ) or ( this.isLocked == 0 ) ) and ( arg1 == "LeftButton" ) then
		this:StartMoving();
		this.isMoving = true;
	end
end

function Gymnast.OnHide()
	if ( this.isMoving ) then
		this:StopMovingOrSizing();
		this.isMoving = false;
	end
end

function Gymnast.OnUpdate()
	if ( this.fadeCount ) then
		this.fadeCount = this.fadeCount - arg1;
		if ( this.fadeCount <= 0 ) then
			this.fadeCount = nil;
			this:FadeOut();
		end
	end
end
