﻿
-- module setup
local me = { name = "gui"}
local mod = thismod
mod[me.name] = me

--[[
Gui.lua

Factory for standard GUI Widgets
]]

me.textinset = 15	-- This is the inset between the edge of the frame and where the text block starts.
me.border = 4		-- This is the experimentally determined width of the borders

--[[
------------------------------------------------------
						Frame
------------------------------------------------------
]]

me.framebackdrop = 
{
	bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
	edgeFile="Interface\\Tooltips\\UI-Tooltip-Border",
	tile = false,
	tileSize = 32,
	edgeSize = 16,
	insets = { left = 4, right = 4, top = 4, bottom = 4 },
}

me.buttonbackdrop = 
{
	bgFile = "Interface\\Tooltips\\UI-Tooltip-Background",
	edgeFile="Interface\\Tooltips\\UI-Tooltip-Border",
	tile = false,
	tileSize = 32,
	edgeSize = 12,
	insets = { left = 3, right = 3, top = 3, bottom = 3 },
}

me.backdropcolor = 
{
	red = 0.2,
	green = 0.2,
	blue = 0.2,
	alpha = 1.0,
}

--[[
mod.gui.tooltip(frame, topic, text)
	Creates a simple tooltip next to a frame.
<frame>	The frame to anchor to, or <this> if none is specified
<topic>	The first line of the tooltip
<text>	The rest of the tooltip
]]
me.tooltip = function(frame, topic, text)
	
	GameTooltip:SetOwner(frame, "ANCHOR_BOTTOM")
	GameTooltip:SetText(topic, 1, 1, 1, 1)
	GameTooltip:AddLine(text, NORMAL_FONT_COLOR.r, NORMAL_FONT_COLOR.g, NORMAL_FONT_COLOR.b, 1)
	GameTooltip:Show()
	
end

--[[
mod.gui.createframe(parent, width, height, usebackground)
	Creates a standard frame with a white border and dark blue background. The frame comes out hidden.

<parent>	frame; if supplied, will set this to be the new frame's parent. Otherwise uses UIParent
<width>	number; the width of the frame, in pixels
<height>	number; the height of the frame, in pixels
<usebackground>	flag; if non-nil the texture bit is added

Return: a Frame object.
]]
me.createframe = function(parent, width, height, usebackground)
	
	if parent == nil then
		parent = UIParent
	end
		
	local frame = CreateFrame("Frame", nil, parent)
	
	-- size
	frame:SetWidth(width)
	frame:SetHeight(height)	
	
	-- backdrop
	frame:SetBackdrop(me.framebackdrop)
	frame:SetBackdropBorderColor(1.0, 1.0, 1.0, 1.0) 
	
	if usebackground then
		frame:SetBackdropColor(me.backdropcolor.red, me.backdropcolor.blue, me.backdropcolor.green, me.backdropcolor.alpha)
	else
		frame:SetBackdropColor(0, 0, 0, 0)
	end
	
	-- we're gonna be greedy and make him high strata
	frame:SetFrameStrata("HIGH")
		
	return frame
	
end

me.createbutton = function(parent, width, height, usebackdrop, fontsize)
	
	if parent == nil then
		parent = UIParent
	end
	
	local button = CreateFrame("Button", nil, parent)
	
	-- size
	button:SetWidth(width)	
	button:SetHeight(height)
	
	-- backdrop
	if usebackdrop then
		button:SetBackdrop(me.buttonbackdrop)
		button:SetBackdropBorderColor(me.backdropcolor.red, me.backdropcolor.blue, me.backdropcolor.green, me.backdropcolor.alpha)
		button:SetBackdropColor(me.backdropcolor.red, me.backdropcolor.blue, me.backdropcolor.green, me.backdropcolor.alpha)
	end
		
	-- font
	if fontsize then
		button:SetFont(me.fontfile, fontsize)
	end
	
	return button
	
end


--[[
------------------------------------------------------
						FontString
------------------------------------------------------
]]

-- This is e.g. "Fonts\\FRIZQT__.TFF" for en/us
me.fontfile = GameFontNormal:GetFont()

--[[
mod.gui.createfontstring(parentframe, fontsize)
	Makes a new fontstring object on the given frame.
	
<parentframe>	Frame; the owner of the label
<fontsize>		number; the pitch of the font

Return:			FontString; a reference to the created fontstring
]]
me.createfontstring = function(parentframe, fontsize, useshadow, flags)
	
	local fontstring = parentframe:CreateFontString()
	fontstring:SetFont(me.fontfile, fontsize, flags)
	
	if useshadow then
		fontstring:SetShadowColor(0, 0, 0)
		fontstring:SetShadowOffset(1, -1)
	end
	
	return fontstring
	
end

--[[
------------------------------------------------------
						Slider
------------------------------------------------------

This is basically a clone of OptionsSliderTemplate from (an old) OptionsFrame.xml, because the CreateFrame isn't working well when inheriting from a virtual template, since it requires a valid part of the global namespace, which we don't want to use.
]]

me.sliderbackdrop = 
{
	bgFile = "Interface\\Buttons\\UI-SliderBar-Background",
	edgeFile="Interface\\Buttons\\UI-SliderBar-Border",
	tile = true,
	tileSize = 8,
	edgeSize = 8,
	insets = { left = 3, right = 3, top = 6, bottom = 6 },
}

--[[
mod.gui.createslider(parent, width, height, fontsize)
	Creates a generic Slider frame.
	
<parent>		Frame; window that owns the slider
<width>		number; width of the slider in screen units (~ pixels)
<height>		number; height of the slider in screen units (~ pixels)
<fontsize>	number; pitch of the labels for the slider

Return:		Slider; reference to the created object
]]
me.createslider = function(parent, width, height, fontsize)
	
	local slider = CreateFrame("Slider", nil, parent)
	
	-- size
	slider:SetWidth(width)
	slider:SetHeight(height)
	
	-- backdrop
	slider:SetBackdrop(me.sliderbackdrop)

	-- the thumb of the slider
	slider:SetThumbTexture("Interface\\Buttons\\UI-SliderBar-Button-Horizontal")
	
	-- orientation
	slider:SetOrientation("HORIZONTAL")
	
	-- labels
	slider.low = me.createfontstring(slider, fontsize)
	slider.low:SetPoint("TOPLEFT", slider, "BOTTOMLEFT", 2, 3)
	
	slider.high = me.createfontstring(slider, fontsize)
	slider.high:SetPoint("TOPRIGHT", slider, "BOTTOMRIGHT", -2, 3)
	slider.high:SetWidth(slider:GetWidth())
	slider.high:SetJustifyH("RIGHT")
	
	slider.text = me.createfontstring(slider, fontsize)
	slider.text:SetPoint("BOTTOMLEFT", slider, "TOPLEFT", 2, 0)
	slider.text:SetTextColor(1.0, 1.0, 0)
	
	slider.value = me.createfontstring(slider, fontsize)
	slider.value:SetPoint("BOTTOMRIGHT", slider, "TOPRIGHT", -2, 0)
	slider.value:SetJustifyH("RIGHT")
	slider.value:SetWidth(slider:GetWidth())
	slider.value:SetTextColor(0.0, 1.0, 0)
	
	-- function to get the actual height
	slider.realheight = function()
		return slider.high:GetHeight() * 2 + slider:GetHeight()
	end
	
	return slider
	
end

me.createscrollframe = function(parent, width, height)
	
	local frame = CreateFrame("ScrollFrame", nil, parent)
	
	-- size
	frame:SetHeight(height)	
	frame:SetWidth(width - 20)
	
	-- backdrop
	frame:SetBackdrop(me.framebackdrop)
	frame:SetBackdropColor(0.0, 0.0, 0.0, 0.0) 
	frame:SetBackdropBorderColor(1.0, 1.0, 1.0, 1.0) 
	
	-- slider (vertical)
	local slider = CreateFrame("Slider", nil, frame)
	frame.slider = slider
	
	-- size
	slider:SetWidth(20)
	slider:SetHeight(height - 40 + 2 * me.border)
	
	-- attach to frame
	slider:SetPoint("TOPLEFT", frame, "TOPRIGHT", 0, -20 + me.border)
	slider:SetPoint("BOTTOMLEFT", frame,  "BOTTOMRIGHT", 0, 20 - me.border)
	
	slider:SetBackdrop(me.sliderbackdrop)
	
	-- range
	slider:SetMinMaxValues(0, 100)
	slider:SetValueStep(1)
	
	-- thumb
	slider:SetThumbTexture("Interface\\Buttons\\UI-ScrollBar-Knob")
	slider:GetThumbTexture():SetTexCoord(0, 1, 0.15, 0.85)
	
	-- scroll up button
	local buttonup = CreateFrame("Button", nil, frame)
	frame.buttonup = buttonup
	
	buttonup:SetWidth(20)
	buttonup:SetHeight(20)
	buttonup:SetPoint("BOTTOM", slider, "TOP", 0, -me.border)
	
	buttonup:SetNormalTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Up")
	buttonup:SetPushedTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Down")
	buttonup:SetDisabledTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Disabled")
	buttonup:SetHighlightTexture("Interface\\Buttons\\UI-ScrollBar-ScrollUpButton-Highlight")
	
	buttonup:GetNormalTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	buttonup:GetPushedTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	buttonup:GetDisabledTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	buttonup:GetHighlightTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	
	-- scroll down button
	local buttondown = CreateFrame("Button", nil, frame)
	frame.buttondown = buttondown
	
	buttondown:SetWidth(20)
	buttondown:SetHeight(20)
	buttondown:SetPoint("TOP", slider, "BOTTOM", 0, me.border)
	
	buttondown:SetNormalTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Up")
	buttondown:SetPushedTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Down")
	buttondown:SetDisabledTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Disabled")
	buttondown:SetHighlightTexture("Interface\\Buttons\\UI-ScrollBar-ScrollDownButton-Highlight")
	
	buttondown:GetNormalTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	buttondown:GetPushedTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	buttondown:GetDisabledTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	buttondown:GetHighlightTexture():SetTexCoord(0.25, 0.75, 0.25, 0.75)
	
	-- click methods
	frame.onclick = function(direction)
		mod.print("clicked " .. direction)
	end
	
	buttonup:SetScript("OnClick", function()
		this:GetParent().onclick("up")
	end)
	
	buttondown:SetScript("OnClick", function()
		this:GetParent().onclick("down")
	end)
	
	return frame
	
end