-- BeneCast by Skurel
-- v 2.1.2
-- Code heavily referenced from Danboo's Healomatic/CastParty. Thank you Danboo!
-- Makes clickable buttons that casts beneficial spells on party members and the current target

-- *****************************************************************************
-- BeneCast Constants
-- *****************************************************************************
-- Maximum number of buttons in a BeneCast Panel
BENECAST_MAX_NUMBER_OF_BUTTONS = 20;
-- This is used to reset the configuration if severely outdated
BENECAST_VERSION = 2.2;

-- *****************************************************************************
-- BeneCast Raid Tables
-- *****************************************************************************
BENECAST_RAID_LIST = {};
BENECAST_RAID_SUBGROUPS = { 0, 0, 0, 0, 0, 0, 0, 0 };
BENECAST_RAID_ROSTER = {}; -- Raid Roster Cache
BENECAST_RAID_ROSTER2 = {}; -- Raid Roster Cache, lookup unit by name
BENECAST_RAID_PANELS = {}; -- panel pool, index = parentframename, value = panelid
BENECAST_RAID_PANELS2 = {}; -- panel pool, index = panelid, value = { unitid = unit, parentframe = parentframename }
BENECAST_RAID_PANELS3 = {}; -- panel pool, index = unitid, value = table of parentframenames

-- BeneCast will hook into CT_RA's Update function to update raid-panels AFTER CT_RA
BC_Old_CTRA_UpdateFunction = nil;

-- BeneCast will only update raidpanels-positions every 5 seconds
BC_UpdateRaid = nil;
BC_Last_UpdateRaid_Done = nil;

-- *****************************************************************************
-- BeneCast UI Tables
-- *****************************************************************************

-- Table to hold all the OptionCheckButton information
-- table value: Text of the checkbutton label
-- index: ID of the checkbutton
-- cvar: Option modified in BeneCastConfig
-- tooltipText: Text for the tooltip of the checkbutton
local BeneCastOptionFrameCheckButtons = {
	TEXT_DAMAGE_BASED_HEALS = { index = 1, cvar = 'DmgBasedHeal', tooltipText = 'TEXT_DAMAGE_BASED_HEALS_TOOLTIP' },
	TEXT_OVERHEAL = { index = 2, cvar = 'Overheal', tooltipText = 'TEXT_OVERHEAL_TOOLTIP' },
	TEXT_SHOWALLBUTTONS = { index = 3, cvar = 'ShowAllButtons', tooltipText = 'TEXT_SHOWALLBUTTONS_TOOLTIP' },
	TEXT_HIDE_MINIMAP_BUTTON = { index = 4, cvar = 'HideMinimap', tooltipText = 'TEXT_HIDE_MINIMAP_BUTTON_TOOLTIP' },
	TEXT_TOOLTIPS = { index = 5, cvar = 'ShowTooltips', tooltipText = 'TEXT_TOOLTIPS_TOOLTIP' },
	TEXT_TOOLTIPS_NAME = { index = 6, cvar = 'ShowTooltipsName', tooltipText = 'TEXT_TOOLTIPS_NAME_TOOLTIP' },
	TEXT_UNLOCK_BUTTONS = { index = 7, cvar = 'Unlock', tooltipText = 'TEXT_UNLOCK_BUTTONS_TOOLTIP' },
	TEXT_NOTIFICATION = { index = 8, cvar = 'Notification', tooltipText = 'TEXT_PARTY_NOTIFICATION_TOOLTIP' },
	TEXT_CASCADE_CHANNELS = { index = 9, cvar = 'CascadeChannels', tooltipText = 'TEXT_CASCADE_CHANNELS_TOOLTIP' },
	TEXT_NOTIFY_FAILURE = { index = 10, cvar = 'NotifyFailure', tooltipText = 'TEXT_NOTIFY_FAILURE_TOOLTIP' },
	TEXT_NOTIFY_RARE = { index = 11, cvar = 'NotifyRare', tooltipText = 'TEXT_NOTIFY_RARE_TOOLTIP' },
	--12 is free
	TEXT_NOTIFY_SELF_CASTS = { index = 13, cvar = 'SelfCasts', tooltipText = 'TEXT_NOTIFY_SELF_CASTS_TOOLTIP' },
	TEXT_NOTIFY_RANK = { index = 14, cvar = 'NotifyRank', tooltipText = 'TEXT_NOTIFY_RANK_TOOLTIP' },
	TEXT_NOTIFY_TIME = { index = 15, cvar = 'NotifyTime', tooltipText = 'TEXT_NOTIFY_TIME_TOOLTIP' },
	TEXT_NOTIFY_HEALS = { index = 16, cvar = 'NotifyHeal', tooltipText = 'TEXT_NOTIFY_HEALS_TOOLTIP' },
	TEXT_NOTIFY_MAX_HEALS = { index = 17, cvar = 'NotifyMaxHeal', tooltipText = 'TEXT_NOTIFY_MAX_HEALS_TOOLTIP' },
	TEXT_NOTIFY_CURES = { index = 18, cvar = 'NotifyCure', tooltipText = 'TEXT_NOTIFY_CURES_TOOLTIP' },
	TEXT_NOTIFY_BUFFS = { index = 19, cvar = 'NotifyBuff', tooltipText = 'TEXT_NOTIFY_BUFFS_TOOLTIP' },
	--20 is free
	TEXT_NOTIFY_RES = { index = 21, cvar = 'NotifyRes', tooltipText = 'TEXT_NOTIFY_RES_TOOLTIP' },
	TEXT_PLAYER_AS_DEFAULT = { index = 22, cvar = 'PlayerAsDefault', tooltipText = 'TEXT_PLAYER_AS_DEFAULT_TOOLTIP' },
	TEXT_SHOW_PETS = { index = 23, cvar = 'ShowPets', tooltipText = 'TEXT_SHOW_PETS_TOOLTIP' },
	TEXT_SHOW_RAID = { index = 24, cvar = 'ShowRaid', tooltipText = 'TEXT_SHOW_RAID_TOOLTIP' },
	TEXT_SHOW_RAID_PETS = { index = 25, cvar = 'ShowRaidPets', tooltipText = 'TEXT_SHOW_RAID_PETS_TOOLTIP' },
	TEXT_REVERSE_HOTS = { index = 26, cvar = 'ReverseHoTs', tooltipText = 'TEXT_REVERSE_HOTS_TOOLTIP' },
	TEXT_MANA_BASED_HEALS = { index = 27, cvar = 'ManaBasedHeal', tooltipText = 'TEXT_MANA_BASED_HEALS_TOOLTIP' },
	TEXT_FLASH_AS_FADE = { index = 28, cvar = 'FlashAsFade', tooltipText = 'TEXT_FLASH_AS_FADE_TOOLTIP' },
	-- 29 is free
	TEXT_CHECK_RANGE = { index = 30, cvar = 'CheckRange', tooltipText = 'TEXT_CHECK_RANGE_TOOLTIP' },
	TEXT_TARGET_OF_TARGET = { index = 31, cvar = 'ShowTargetTarget', tooltipText = 'TEXT_TARGET_OF_TARGET_TOOLTIP' },
};


-- Table to hold all the OptionCheckButton information
-- text: Text of the slider label
-- index: ID of the slider
-- cvar: Option modified in BeneCastConfig
-- minValue: Minimum value of the slider
-- maxValue: Maximum value of the slider
-- valueStep: Step value of the slider when being changed
-- tooltipText: Text for the tooltip of the checkbutton
local BeneCastOptionFrameSliders = {
	{ index = 1, text = 'TEXT_BUTTON_SIZE', cvar = 'ButtonSize', minValue = 8, maxValue = 36, valueStep = 1, tooltipText = 'TEXT_BUTTONSIZESLIDER_TOOLTIP' },
	{ index = 2, text = 'TEXT_RANKS_TO_OVERHEAL', cvar = 'OverhealSlider', minValue = 0, maxValue = 10, valueStep = 1, tooltipText = 'TEXT_OVERHEALSLIDER_TOOLTIP' },
	{ index = 3, text = 'TEXT_BUTTON_NUMBER', cvar = 'ButtonNumber', minValue = 1, maxValue = 20, valueStep = 1, tooltipText = 'TEXT_BUTTONNUMBERSLIDER_TOOLTIP' },
};

local BeneCastOptionDropDowns = {
	{ index = 1, text = 'TEXT_NOTIFY_SELF_CASTS', cvar = 'SelfCastsChannel', tooltipText = 'TEXT_NOTIFY_SELF_CASTS_TOOLTIP', enable = 'SelfCasts' },
	{ index = 2, text = 'TEXT_NOTIFY_HEALS', cvar = 'NotifyHealChannel', tooltipText = 'TEXT_NOTIFY_HEALS_TOOLTIP', enable = 'NotifyHeal' },
	{ index = 3, text = 'TEXT_NOTIFY_CURES', cvar = 'NotifyCureChannel', tooltipText = 'TEXT_NOTIFY_CURES_TOOLTIP', enable = 'NotifyCure' },
	{ index = 4, text = 'TEXT_NOTIFY_BUFFS', cvar = 'NotifyBuffChannel', tooltipText = 'TEXT_NOTIFY_BUFFS_TOOLTIP', enable = 'NotifyBuff' },
	{ index = 5, text = 'TEXT_NOTIFY_RES', cvar = 'NotifyResChannel', tooltipText = 'TEXT_NOTIFY_RES_TOOLTIP', enable = 'NotifyRes' },
	{ index = 6, text = 'TEXT_NOTIFY_FAILURE', cvar = 'NotifyFailureChannel', tooltipText = 'TEXT_NOTIFY_FAILURE_TOOLTIP', enable = 'NotifyFailure' },
	{ index = 7, text = 'TEXT_NOTIFY_RARE', cvar = 'NotifyRareChannel', tooltipText = 'TEXT_NOTIFY_RARE_TOOLTIP', enable = 'NotifyRare' },
};

local BeneCastNotificationTargets = {
	TEXT_SELF_CHANNEL = 'SELF',
	TEXT_PARTY_CHANNEL = 'PARTY',
	TEXT_RAID_CHANNEL = 'RAID',
	TEXT_WHISPER_TARGET = 'WHISPER',
	TEXT_EMOTE_CHANNEL = 'EMOTE',
	TEXT_SAY_CHANNEL = 'SAY',
	TEXT_NOTIFY_USER_CHANNEL = 'USER',
};

-- Table to hold the different subframe names in the BeneCastOptions frame
local BeneCastOptionFrameSubframes = {'BeneCastPlayerFrame', 'BeneCastClass1Frame', 'BeneCastClass2Frame', 'BeneCastClass3Frame', 'BeneCastClass4Frame', 'BeneCastClass5Frame', 'BeneCastClass6Frame', 'BeneCastClass7Frame', 'BeneCastClass8Frame', 'BeneCastClass9Frame', 'BeneCastSetupFrame', 'BeneCastNotificationFrame', 'BeneCastRaidFrame', 'BeneCastSnapToFrame', };

-- *****************************************************************************
-- Global variables
-- *****************************************************************************

-- Array to hold possible targets
local BC_targets = { 'player', 'party1', 'party2', 'party3', 'party4', 'raid1', 'raid2', 'raid3', 'raid4', 'raid5', 'raid6', 'raid7', 'raid8', 'raid9', 'raid10', 'raid11', 'raid12', 'raid13', 'raid14', 'raid15', 'raid16', 'raid17', 'raid18', 'raid19', 'raid20', 'raid21', 'raid22', 'raid23', 'raid24', 'raid25', 'raid26', 'raid27', 'raid28', 'raid29', 'raid30', 'raid31', 'raid32', 'raid33', 'raid34', 'raid35', 'raid36', 'raid37', 'raid38', 'raid39', 'raid40', 'pet', 'partypet1', 'partypet2', 'partypet3', 'partypet4', 'raidpet1', 'raidpet2', 'raidpet3', 'raidpet4', 'raidpet5', 'raidpet6', 'raidpet7', 'raidpet8', 'raidpet9', 'raidpet10', 'raidpet11', 'raidpet12', 'raidpet13', 'raidpet14', 'raidpet15', 'raidpet16', 'raidpet17', 'raidpet18', 'raidpet19', 'raidpet20', 'raidpet21', 'raidpet22', 'raidpet23', 'raidpet24', 'raidpet25', 'raidpet26', 'raidpet27', 'raidpet28', 'raidpet29', 'raidpet30', 'raidpet31', 'raidpet32', 'raidpet33', 'raidpet34', 'raidpet35', 'raidpet36', 'raidpet37', 'raidpet38', 'raidpet39', 'raidpet40', 'target', 'targettarget', 'focus' };

-- Array to hold the list of targets we are currently attached to
local BC_attached_targets = {};

--WINTROW.6 Array to hold possible endings for EditBoxes
local BCW_components = { 'Frame', 'Point', 'RelativePoint', 'X', 'Y', };

-- Array to assign id numbers based on class
local BC_classes = {
	[BENECAST_STRINGS.CLASS_DRUID]		= 2,
	[BENECAST_STRINGS.CLASS_MAGE]		= 4,
	[BENECAST_STRINGS.CLASS_HUNTER]		= 3,
	[BENECAST_STRINGS.CLASS_PRIEST]		= 5,
	[BENECAST_STRINGS.CLASS_ROGUE]		= 6,
	[BENECAST_STRINGS.CLASS_WARLOCK]	= 7,
	[BENECAST_STRINGS.CLASS_WARRIOR]	= 8,
	[BENECAST_STRINGS.CLASS_SHAMAN]		= 9,
	[BENECAST_STRINGS.CLASS_PALADIN]	= 10,
}

-- Table used to sort spelltypes
local BC_spellsort = {
	'res', 'efficient', 'efficient2', 'efficient3', 'emergency', 'instant', 'instant2', 'loh', 'group', 'group2',
	'poison', 'disease', 'disease2', 'magic', 'curse', 'buff1', 'buff2', 'buff3', 'buff4', 'partybuff1', 'buff5',
	'buff6', 'buff7', 'buff8', 'buff9', 'buff10', 'groupbuff1', 'groupbuff2', 'groupbuff3', 'buffparty1g',
	'buffparty2g', 'buffparty3g', 'buffparty4g', 'buffparty5g', 'buffparty6g', 'buffparty7g', 'buffparty8g',
	'buffparty9g', 'selfbuff1', 'selfbuff2', 'selfbuff3', 'selfbuff4', 'selfbuff5', 'selfbuff6', 'selfbuff7',
	'selfbuff8', 'selfbuff9', 'selfbuff10', 'selfbuff11', 'selfbuff12', 'selfbuff13', 'weaponenchant1',
	'weaponenchant2', 'weaponenchant3', 'weaponenchant4', 'partybuff2', 'partybuff3', 'partybuff4',
	'partybuff5', 'partybuff6', 'partybuff7', 'partybuff8', 'partybuff9', 'partybuff10', 'firstaid'
};

-- Table used to determine the buffs in effect in UpdateButtons
BC_buffs_in_effect = {};
BC_buffs_in_effect[1] = { fade = nil, name = 'buff1', notabuff = false };
BC_buffs_in_effect[2] = { fade = nil, name = 'buff2', notabuff = false };
BC_buffs_in_effect[3] = { fade = nil, name = 'buff3', notabuff = false };
BC_buffs_in_effect[4] = { fade = nil, name = 'buff4', notabuff = false };
BC_buffs_in_effect[5] = { fade = nil, name = 'buff5', notabuff = false };
BC_buffs_in_effect[6] = { fade = nil, name = 'buff6', notabuff = false };
BC_buffs_in_effect[7] = { fade = nil, name = 'groupbuff1', notabuff = false };
BC_buffs_in_effect[8] = { fade = nil, name = 'buffparty1g', notabuff = false };
BC_buffs_in_effect[9] = { fade = nil, name = 'buffparty2g', notabuff = false };
BC_buffs_in_effect[10] = { fade = nil, name = 'buffparty3g', notabuff = false };
BC_buffs_in_effect[11] = { fade = nil, name = 'buffparty4g', notabuff = false };
BC_buffs_in_effect[12] = { fade = nil, name = 'buffparty5g', notabuff = false };
BC_buffs_in_effect[13] = { fade = nil, name = 'buffparty6g', notabuff = false };
BC_buffs_in_effect[14] = { fade = nil, name = 'buffparty7g', notabuff = false };
BC_buffs_in_effect[15] = { fade = nil, name = 'buffparty8g', notabuff = false };
BC_buffs_in_effect[16] = { fade = nil, name = 'buffparty9g', notabuff = false };
BC_buffs_in_effect[17] = { fade = nil, name = 'selfbuff1', notabuff = false };
BC_buffs_in_effect[18] = { fade = nil, name = 'selfbuff2', notabuff = false };
BC_buffs_in_effect[19] = { fade = nil, name = 'selfbuff3', notabuff = false };
BC_buffs_in_effect[20] = { fade = nil, name = 'selfbuff4', notabuff = false };
BC_buffs_in_effect[21] = { fade = nil, name = 'selfbuff5', notabuff = false };
BC_buffs_in_effect[22] = { fade = nil, name = 'selfbuff6', notabuff = false };
BC_buffs_in_effect[23] = { fade = nil, name = 'selfbuff7', notabuff = false };
BC_buffs_in_effect[24] = { fade = nil, name = 'selfbuff8', notabuff = false };
BC_buffs_in_effect[25] = { fade = nil, name = 'selfbuff9', notabuff = false };
BC_buffs_in_effect[26] = { fade = nil, name = 'selfbuff10', notabuff = false };
BC_buffs_in_effect[27] = { fade = nil, name = 'selfbuff11', notabuff = false };
BC_buffs_in_effect[28] = { fade = nil, name = 'selfbuff12', notabuff = false };
BC_buffs_in_effect[29] = { fade = nil, name = 'selfbuff13', notabuff = false };
BC_buffs_in_effect[30] = { fade = nil, name = 'groupbuff2', notabuff = false };
BC_buffs_in_effect[31] = { fade = nil, name = 'groupbuff3', notabuff = false };
BC_buffs_in_effect[32] = { fade = nil, name = 'buff7', notabuff = false };
BC_buffs_in_effect[33] = { fade = nil, name = 'buff8', notabuff = false };
BC_buffs_in_effect[34] = { fade = nil, name = 'buff9', notabuff = false };
BC_buffs_in_effect[35] = { fade = nil, name = 'buff10', notabuff = false };
BC_buffs_in_effect[36] = { fade = nil, name = 'partybuff1', notabuff = false };
BC_buffs_in_effect[37] = { fade = nil, name = 'partybuff2', notabuff = false };
BC_buffs_in_effect[38] = { fade = nil, name = 'partybuff3', notabuff = false };
BC_buffs_in_effect[39] = { fade = nil, name = 'partybuff4', notabuff = false };
BC_buffs_in_effect[40] = { fade = nil, name = 'partybuff5', notabuff = false };
BC_buffs_in_effect[41] = { fade = nil, name = 'partybuff6', notabuff = false };
BC_buffs_in_effect[42] = { fade = nil, name = 'partybuff7', notabuff = false };
BC_buffs_in_effect[43] = { fade = nil, name = 'partybuff8', notabuff = false };
BC_buffs_in_effect[44] = { fade = nil, name = 'partybuff9', notabuff = false };
BC_buffs_in_effect[45] = { fade = nil, name = 'partybuff10', notabuff = false };
BC_buffs_in_effect[46] = { fade = nil, name = 'instant', notabuff = true };
BC_buffs_in_effect[47] = { fade = nil, name = 'emergency', notabuff = true };
BC_buffs_in_effect[48] = { fade = nil, name = 'poison', notabuff = true };

local BC_buffs_in_effect_by_name = {};

-- Table to hold buttons by spelltype
local BC_buttons_by_spell = {};
local BC_buttons_that_flash = {};
local BC_LastRangeCheck = nil;

-- Healing threshold from max
local BC_threshold = 1;

-- Class of the player
local BC_class;

-- Party notification stuff
local BC_spellbeingcast = nil;
local BC_spelltarget = nil;
local BC_spellismax = nil;
local BC_spellstarted = nil;
local BC_spellstopped = nil;
local BC_sentmessage = nil;

-- Weapon Enchant status
local BC_weaponenchant = nil;

-- Statusses of current player
local BC_clearcasting = nil;
local BC_spirit_of_redemption = nil;
local BC_innerfocus = nil;

local BC_AttachPartyFramesOnTargetChange = nil;
local BC_MarkedTCTable = nil;
local BC_ConsiderInCombat = nil;

local BC_BonusScanner_HookFunction = nil;

-- Optionscreen stuff
local BC_PanelNumber = 1;
local BC_CurrentProfile = nil;

-- Plus Healing modifier
local BC_plusheal = 0;

-- Table of equipment to scan for Plus Healing modifiers
local BC_equipslots = {
	"Head",
	"Neck",
	"Shoulder",
	"Shirt",
	"Chest",
	"Waist",
	"Legs",
	"Feet",
	"Wrist",
	"Hands",
	"Finger0",
	"Finger1",
	"Trinket0",
	"Trinket1",
	"Back",
	"MainHand",
	"SecondaryHand",
	"Ranged",
	"Tabard",
};

-- String for checking if a line in an equipment slot is part of a set
local BC_setstring = '^(.*) %(%d/%d%)$';

-- Tables for interpolating health of targets outside the group
local BC_class_health = {
	[BENECAST_STRINGS.CLASS_DRUID]	= { ['1'] = 30, ['60'] = 3000 },
	[BENECAST_STRINGS.CLASS_MAGE]	= { ['1'] = 30, ['60'] = 3000 },
	[BENECAST_STRINGS.CLASS_HUNTER]	= { ['1'] = 30, ['60'] = 4000 },
	[BENECAST_STRINGS.CLASS_PALADIN]= { ['1'] = 30, ['60'] = 4500 },
	[BENECAST_STRINGS.CLASS_PRIEST]	= { ['1'] = 30, ['60'] = 3000 },
	[BENECAST_STRINGS.CLASS_ROGUE]	= { ['1'] = 30, ['60'] = 3500 },
	[BENECAST_STRINGS.CLASS_SHAMAN]	= { ['1'] = 30, ['60'] = 4000 },
	[BENECAST_STRINGS.CLASS_WARLOCK]= { ['1'] = 30, ['60'] = 4000 },
	[BENECAST_STRINGS.CLASS_WARRIOR]= { ['1'] = 30, ['60'] = 5000 },
};

-- Storage for spell data
local BC_spell_data = {};

-- Table to hold data of the Target of Target
local BC_TargetTarget = {};

-- *****************************************************************************
-- BeneCast Helper functions
-- *****************************************************************************

-- Function to clear the table given as the parameter
function BeneCast_ClearTable(Table)

	-- Removes every element of the table starting at the end
	while table.getn(Table) > 0 do
		table.remove(Table);
	end
	
end

-- *****************************************************************************
-- BeneCast Option Helper functions
-- *****************************************************************************

-- Function to set orientation of buttons
function BeneCast_OrientButtons(id)
	
	if id == nil then
		PutDebugMsg('BeneCast_OrientButtons(): id is nil');
		return;
	end
	
	if not BeneCastConfig['BeneCastPanel' .. id] then
		BeneCastConfig['BeneCastPanel' .. id] = {};
	end
	
	if not BeneCastConfig['BeneCastPanel' .. id]['Orientation'] then
		BeneCastConfig['BeneCastPanel' .. id]['Orientation'] = 'right';
	end
	
	local previousbutton = getglobal('BeneCastPanel' .. id .. 'Button1');
	if InCombatLockdown() then
		return;
	end
	local lastvisiblebutton = 0;
	
	if previousbutton:IsVisible() then
		lastvisiblebutton = 1;
	end
	
	for j = 2, BENECAST_MAX_NUMBER_OF_BUTTONS do
		-- The first button is fine where it is
		local button = getglobal('BeneCastPanel' .. id .. 'Button' .. j);
		
		button:ClearAllPoints();
		if BeneCastConfig['BeneCastPanel' .. id]['Orientation'] == 'down' then
			button:SetPoint('TOPLEFT',previousbutton,'BOTTOMLEFT',0,-3);
		else
			button:SetPoint('TOPLEFT',previousbutton,'TOPRIGHT',3,0);
		end
		
		if button:IsVisible() then
			lastvisiblebutton = j;
		end
		
		previousbutton = button;
	end
	
	BeneCast_ResizeButtonsFor(id);

end

-- Function to resize the buttons of one panel
function BeneCast_ResizeButtonsFor(i)

	local buttonsize = BeneCastConfig.ButtonSize;
	local button, buttonborder, buttoncooldown, buttonfade, frame, buttonrange;
	frame = getglobal('BeneCastPanel' .. i);
	if InCombatLockdown() then
		return;
	end

	-- Resize the buttons based on the saved size for every button in a panel
	for j = 1,BENECAST_MAX_NUMBER_OF_BUTTONS do
		button = getglobal('BeneCastPanel' .. i .. 'Button' .. j);
		-- Set Button height and width to buttonsize
		button:SetHeight(buttonsize);
		button:SetWidth(buttonsize);
		buttonborder = getglobal('BeneCastPanel' .. i .. 'Button' .. j .. 'NormalTexture');
		-- Set the Button border to fit around the button
		buttonborder:SetHeight(buttonsize * 2 - (buttonsize/36) * 10);
		buttonborder:SetWidth(buttonsize * 2 - (buttonsize/36) * 10);
		buttoncooldown = getglobal('BeneCastPanel' .. i .. 'Button' .. j .. 'Cooldown');
		-- Set Cooldown size
		buttoncooldown:SetScale((buttonsize / 36));
		buttoncooldown:SetPoint('CENTER', button, 'CENTER');
		buttonfade = getglobal('BeneCastPanel' .. i .. 'Button' .. j .. 'Fade');
		-- Set Fade size
		buttonfade:SetHeight(buttonsize * 2 - (buttonsize/36) * 10);
		buttonfade:SetWidth(buttonsize * 2 - (buttonsize/36) * 10);
		buttonrange = getglobal('BeneCastPanel' .. i .. 'Button' .. j .. 'Range');
		-- Set Range size
		buttonrange:SetHeight(buttonsize * 2 - (buttonsize/36) * 10);
		buttonrange:SetWidth(buttonsize * 2 - (buttonsize/36) * 10);
		-- Resize the BeneCastPanelFrame based on the number of buttons shown
		if ( button:IsVisible() ) then
			if not BeneCastConfig['BeneCastPanel' .. i] then
				BeneCastConfig['BeneCastPanel' .. i] = {};
			end
			frame = getglobal('BeneCastPanel' .. i);
			if BeneCastConfig['BeneCastPanel' .. i]['Orientation'] == 'down' then
				frame:SetHeight(buttonsize * j + 11 + (3 * (j - 1)));
				frame:SetWidth(buttonsize + 11);
			else
				frame:SetWidth(buttonsize * j + 11 + (3 * (j - 1)));
				frame:SetHeight(buttonsize + 11);
			end
		end
	end

end

-- Function to resize the buttons
function BeneCast_ResizeButtons()

	-- Resize the buttons for every unit in BC_targets
	-- i is the number associated with each unit	
	for i in pairs (BC_targets) do		
		BeneCast_ResizeButtonsFor(i);
	end
	
end


-- Function to load up the saved values in BeneCastConfig
function BeneCast_LoadOptions()

	-- Load the String Data and saved value into the check buttons
	local button;
	local string;
	local checked;
	for index, value in pairs (BeneCastOptionFrameCheckButtons) do
		button = getglobal('BeneCastOptionCheckButton'..value.index);
		string = getglobal('BeneCastOptionCheckButton'..value.index..'Text');
		checked = BeneCastConfig[value.cvar];
		button:SetChecked(checked);
		string:SetText(BENECAST_STRINGS[index]);
		button.tooltipText = BENECAST_STRINGS[value.tooltipText];
		-- If this is the tooltip option check to see if it's enabled
		-- If not disable the tooltip name option checkbutton
		if ( value.index == 5 and not checked ) then
			getglobal('BeneCastOptionCheckButton6'):Disable();
		-- If this is the notification option check to see if it's enabled
		-- If not disable all sub notification options
		elseif ( value.index == 8 and not checked ) then
			getglobal('BeneCastOptionCheckButton9'):Disable();
			getglobal('BeneCastOptionCheckButton10'):Disable();
			getglobal('BeneCastOptionCheckButton11'):Disable();
			getglobal('BeneCastOptionCheckButton13'):Disable();
			getglobal('BeneCastOptionCheckButton14'):Disable();
			getglobal('BeneCastOptionCheckButton15'):Disable();
			getglobal('BeneCastOptionCheckButton16'):Disable();
			getglobal('BeneCastOptionCheckButton17'):Disable();
			getglobal('BeneCastOptionCheckButton18'):Disable();
			getglobal('BeneCastOptionCheckButton19'):Disable();
			getglobal('BeneCastOptionCheckButton21'):Disable();
		-- If this is the heal notification option check to see if it's enabled
		-- If not disable all heal max notification
		elseif ( value.index == 16 and not checked ) then
			getglobal('BeneCastOptionCheckButton17'):Disable();
		end
	end

	local slider;
	local string;
	local curvalue;
	-- Load the String Data and saved value into the sliders
	for index, value in pairs (BeneCastOptionFrameSliders) do
		slider = getglobal('BeneCastOptionSlider'..value.index);
		string = getglobal('BeneCastOptionSlider'..value.index..'Text');
		curvalue = getglobal('BeneCastOptionSlider'..value.index..'Value');
		slider:SetMinMaxValues(value.minValue, value.maxValue);
		slider:SetValueStep(value.valueStep);
		slider:SetValue(BeneCastConfig[value.cvar]);
		curvalue:SetText(BeneCastConfig[value.cvar]);
		string:SetText(BENECAST_STRINGS[value.text]);
		slider.tooltipText = BENECAST_STRINGS[value.tooltipText];
	end
	
	-- Load the user defined chat channel
	if ( BeneCastConfig['UserChannel'] ) then
		getglobal('BeneCastChannelEditBox'):SetText(BeneCastConfig['UserChannel']);
	end
	
	BeneCastUpdateCustomEditBoxes();
end

function BeneCastUpdateCustomEditBoxes()
	local editboxframe = getglobal('BeneCastPanelNumber');
	editboxframe:SetText(BC_PanelNumber);
	local i = BC_PanelNumber;
	
	for j, component in pairs (BCW_components) do
		editboxframe = getglobal('BeneCastPanel' .. component);
		if editboxframe then
			if BeneCastConfig['BeneCastPanel' .. i] then
				if BeneCastConfig['BeneCastPanel' .. i][component] ~= nil then
					editboxframe:SetText(BeneCastConfig['BeneCastPanel' .. i][component]);
				else
					editboxframe:SetText('nil');
				end
			end
		end
	end
end

-- Loads the data into the spell check buttons
function BeneCast_LoadSpellConfig()

	-- Position of the current checkbuttons to label
	local i = 1;
	
	-- Name of the spell to make a checkbutton for
	local spellname;
	
	-- Assist checkbutton
	for j = 2,10 do
		getglobal( BeneCastOptionFrameSubframes[j] .. 'Button' .. i .. 'Text' ):SetText(BENECAST_STRINGS.TEXT_ASSIST);
		getglobal( BeneCastOptionFrameSubframes[j] .. 'Button' .. i ):Show();
		getglobal( BeneCastOptionFrameSubframes[j] .. 'Button' .. i ):SetChecked(BeneCastConfig[j]['assist']);
	end
	
	i = i + 1;
	
	-- Load up the spell buttons in the order in BC_spellsort
	for j, spelltype in pairs (BC_spellsort) do
		-- If this class can cast this spelltype continue
		if BC_spell_data[spelltype] then
			local k = table.getn(BeneCast_SpellTypes_Sorting[BC_class][spelltype]);
			if k == nil then
				PutMsg('Can not find ' .. spelltype .. ' for class ' .. BC_class .. ' in Sortingtable !');
			else
				spellname = BeneCast_SpellTypes_Sorting[BC_class][spelltype][k];
				-- If this is a selfbuff only add it to the player frame
				local selfbuff = nil;
				if string.find(spelltype, 'self') then
					selfbuff = true;
				elseif ( ( BC_class == BENECAST_STRINGS.CLASS_HUNTER ) or ( BC_class == BENECAST_STRINGS.CLASS_WARLOCK ) and
				         spelltype == 'efficient' ) then
					selfbuff = true;
				elseif string.find(spelltype, 'weapon') then
					selfbuff = true;
				end
				if selfbuff then
					getglobal( BeneCastOptionFrameSubframes[1] .. 'Button' .. i .. 'Text' ):SetText(spellname);
					getglobal( BeneCastOptionFrameSubframes[1] .. 'Button' .. i ):Show();
					getglobal( BeneCastOptionFrameSubframes[1] .. 'Button' .. i ):SetChecked(BeneCastConfig[1][spelltype]);
				-- Otherwise add it to every spell subframe
				else
					for k = 1,10 do
						getglobal( BeneCastOptionFrameSubframes[k] .. 'Button' .. i .. 'Text' ):SetText(spellname);
						getglobal( BeneCastOptionFrameSubframes[k] .. 'Button' .. i ):Show();
						getglobal( BeneCastOptionFrameSubframes[k] .. 'Button' .. i ):SetChecked(BeneCastConfig[k][spelltype]);
					end
				end
				-- Increment i if a spell was added to the subframes
				i = i + 1;
			end
		end
	end
end

-- Resets data tables
function BeneCast_ResetData()

	-- Clear the BC_spell_data table
	for type in pairs (BC_spell_data) do
		BeneCast_ClearTable(BC_spell_data[type]['spells']);
	end
	
	-- Reload the spell data
	BeneCast_LoadSpellData();
	
	-- Reload the spell checkbuttons
	BeneCast_LoadSpellConfig();
	
end

-- *****************************************************************************
-- BeneCast Frame Attachment Helper functions
-- *****************************************************************************

--Function to do several attaches at once
function BeneCast_AttachPartyPanelsFromVar()

	--players
	for x = 1, 5 do
		BeneCast_AttachPanelsFromVar(x);
	end
	--pets
	for x = 46, 50 do
		BeneCast_AttachPanelsFromVar(x);
	end
	--target
	BeneCast_AttachPanelsFromVar(91);
	--targettarget
	BeneCast_AttachPanelsFromVar(92);
	--focus
	BeneCast_AttachPanelsFromVar(93);

	
end

-- Function to attach BeneCast panels to unit frames
-- Not raid frames, raid frames are handled with BeneCast_AttachRaidPanels
function BeneCast_SnapPanels(AddOnFrames) 

	if ( not AddOnFrames ) then
		PutDebugMsg('AddOnFrames is nil');
		return;
	end
	
	PutDebugMsg('BeneCast_SnapPanels(' .. AddOnFrames .. ')');
	
	local snapto_option = nil;
	
	for temp_i in pairs (BeneCast_SnapTo) do
		if ( BeneCast_SnapTo[temp_i].AddOn == AddOnFrames ) then
			PutDebugMsg('Found match for ' .. temp_i);
			snapto_option = temp_i;
			break;
		else
			PutDebugMsg('No match for ' .. temp_i);
		end
	end
	
	if snapto_option then
		PutDebugMsg('Ended up with ' .. snapto_option);
	else
		PutDebugMsg('Nothing found');
		return;
	end;
	
	if ( BeneCastConfig.SnapTo ) then
		if snapto_option ~= BeneCastConfig.SnapTo then
			PutDebugMsg(snapto_option .. ' ~= Current SnapTo option ' .. BeneCastConfig.SnapTo);
			return;
		end
	else
		PutDebugMsg('CheckOptionValue is true and BeneCastConfig.SnapTo is nil');
		return;
	end
	
	for framename, frametable in pairs (BeneCast_SnapTo[snapto_option]) do
		if ( frametable and framename ~= 'AddOn' and framename ~= '' ) then
			PutDebugMsg('Reading config for BeneCast_SnapTo[' .. snapto_option .. '][' .. framename .. '] = ' .. frametable.frame);
			if BeneCastConfig[framename] == nil then
				BeneCastConfig[framename] = {};
			end;
			BeneCastConfig[framename].Moved = false;
			BeneCastConfig[framename].Frame = frametable.frame;
			BeneCastConfig[framename].Point = frametable.point;
			BeneCastConfig[framename].RelativePoint = frametable.relativePoint;
			BeneCastConfig[framename].X = frametable.x;
			BeneCastConfig[framename].Y = frametable.y;
			BeneCastConfig[framename].Orientation = frametable.orientation;
		end
	end
	
	--PutDebugMsg('Attaching panels 1-5, 46-50 and 91');
	--BeneCast_AttachPartyPanelsFromVar();
	
	PutDebugMsg('Updating Config Controls');
	
	BeneCastUpdateCustomEditBoxes();
end

function BeneCast_AttachPanelsFromVar(id)
	PutDebugMsg('BeneCast_AttachPanelsFromVar(), entered function');
	if ( not id ) then
		PutDebugMsg('BeneCast_AttachPanelsFromVar(), id is nil');
		return;
	end
	if ( id < 1 ) or ( id > 93 ) then
		PutDebugMsg('BeneCast_AttachPanelsFromVar(), id is out of range (' .. id .. ')');
		return;
	end
	local framename = 'BeneCastPanel' .. id;
	local panel = getglobal(framename);
	if BeneCastConfig[framename] == nil then
		BeneCastConfig[framename] = {
			['Frame'] = '',
			['Point'] = '',
			['RelativePoint'] = '',
			['X'] = 0,
			['Y'] = 0,
			['Moved'] = false,
			['Orientation'] = 'right',
		};
	end
	if ( BeneCastConfig[framename]['Frame'] == nil ) or ( BeneCastConfig[framename]['Frame'] == '' ) then
		PutDebugMsg('BeneCast_AttachPanelsFromVar(), config-frame is nil or empty for Panel ' .. id);
		return;
	end
	local parent = getglobal(BeneCastConfig[framename]['Frame']);
	if ( not parent ) then
		PutDebugMsg('BeneCast_AttachPanelsFromVar(), frame ' .. BeneCastConfig[framename]['Frame'] .. ' not found');
		panel:Hide();
		return;
	end

	if ( not BeneCastConfig[framename]['Point'] ) or (BeneCastConfig[framename]['Point'] == '' ) then
		BeneCastConfig[framename]['Point'] = 'TOPLEFT';
	end
	if ( not BeneCastConfig[framename]['RelativePoint'] ) or ( BeneCastConfig[framename]['RelativePoint'] == '' ) then
		BeneCastConfig[framename]['RelativePoint'] = 'TOPRIGHT';
	end
	if ( not BeneCastConfig[framename]['X'] ) then
		BeneCastConfig[framename]['X'] = 0;
	end
	if ( not BeneCastConfig[framename]['Y'] ) then
		BeneCastConfig[framename]['Y'] = 0;
	end
	-- Make sure we got nice rounded numbers
	BeneCastConfig[framename]['X'] = math.floor(BeneCastConfig[framename]['X'] + 0.5);
	BeneCastConfig[framename]['Y'] = math.floor(BeneCastConfig[framename]['Y'] + 0.5);
	if InCombatLockdown() then
		return;
	end
	panel:SetParent(parent);
	panel:ClearAllPoints();
	local scaled_x = BeneCastConfig[framename]['X'] / panel:GetEffectiveScale();
	local scaled_y = BeneCastConfig[framename]['Y'] / panel:GetEffectiveScale();
	panel:SetPoint(BeneCastConfig[framename]['Point'], parent, BeneCastConfig[framename]['RelativePoint'],
	               scaled_x, scaled_y);
	panel:SetFrameStrata(parent:GetFrameStrata());
	panel:SetFrameLevel(parent:GetFrameLevel() + 1);
	-- Set up the correct level for the buttons on the panel
	for x = 1, BENECAST_MAX_NUMBER_OF_BUTTONS do
		getglobal(framename .. 'Button' .. x):SetFrameLevel(panel:GetFrameLevel() + 1);
		getglobal(framename .. 'Button' .. x .. 'Cooldown'):SetFrameLevel(panel:GetFrameLevel() + 2);
	end
	PutDebugMsg('Hanged Panel ' .. framename .. '-s ' .. BeneCastConfig[framename]['Point'] .. '  to ' .. BeneCastConfig[framename]['Frame'] ..
	  '-s ' .. BeneCastConfig[framename]['RelativePoint']);
	
	BeneCast_OrientButtons(id);
end

--Hook function to intercept updates to CT_RA's update function
function BC_UpdateRaidGroup(updateType)
	--First make CT_RA update it's own frames by calling it's own function
	if BC_Old_CTRA_UpdateFunction ~= nil then
		BC_Old_CTRA_UpdateFunction(updateType);
	end
	--Then call our own update-function
	if updateType == 0 or updateType == 3 then
		BC_UpdateRaid = true;
	end
end

--Function to attach BeneCast panels to raid unit frames
function BeneCast_AttachRaidPanels(snapto)

	local name, rank, subgroup, level, class, fileName, zone, online, isDead;
	
	-- If Raid buttons are not being shown don't spend time attaching frames
	if not BeneCastConfig.ShowRaid then
		PutDebugMsg('BeneCast_AttachRaidPanels(): return because Show Raid is disabled');
		return;
	end
	
	if not snapto then
		PutDebugMsg('BeneCast_AttachRaidPanels(): return because snapto is nil/false');
		return;
	end
	
	if not BeneCast_RaidSnapTo[snapto] then
		PutDebugMsg('BeneCast_AttachRaidPanels(): return because BeneCast_RaidSnapTo[' .. snapto .. '] is nil/false');
		return;
	end
	
	if not BeneCast_RaidSnapTo[snapto]['Funct'] then
		PutDebugMsg('BeneCast_AttachRaidPanels(): return because BeneCast_RaidSnapTo[' .. snapto .. '].Func is nil/false');
		return;
	end
	
	if InCombatLockdown() then
		return;
	end
	
	-- Only call function once, it will now do it all in one loop, no parameters needed anymore
	BeneCast_RaidSnapTo[snapto]['Funct']();

end

-- *****************************************************************************
-- BeneCast Raid Helper functions
-- *****************************************************************************

function BeneCast_ClearRaid()

	if InCombatLockdown() then
		return;
	end
	
	-- Reset all the buttons
	for i = 1, MAX_RAID_MEMBERS do
		getglobal('BeneCastRaidButton' .. i):SetChecked('false');
		getglobal('BeneCastRaidButton' .. i):Disable();
		getglobal('BeneCastRaidButton' .. i .. 'Empty'):SetText(BENECAST_STRINGS.TEXT_EMPTY);
		getglobal('BeneCastRaidButton' .. i .. 'Empty'):SetTextColor(GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b);
		getglobal('BeneCastRaidButton' .. i .. 'Name'):SetText('');
		getglobal('BeneCastRaidButton' .. i .. 'Level'):SetText('');
		getglobal('BeneCastRaidButton' .. i .. 'Class'):SetText('');
	end
	
	--Reset Panel Pool too
	BENECAST_RAID_PANELS = {}; 
	BENECAST_RAID_PANELS2 = {};
	BENECAST_RAID_PANELS3 = {};
	
	for i = 6, 45 do
		local panel = getglobal('BeneCastPanel' .. i);
		if panel then
			panel:Hide();
		end
	end
	for i = 51, 90 do
		local panel = getglobal('BeneCastPanel' .. i);
		if panel then
			panel:Hide();
		end
	end
end

function BeneCast_UpdateRaid()

	if not BeneCastConfig.ShowRaid then
		local testtext = getglobal('BeneCastRaidButton1Name'):GetText();
		if testtext ~= nil and testtext ~= '' then
			BeneCast_ClearRaid();
		end
		return;
	end;

	local name, rank, subgroup, level, class, fileName, zone, online, isDead;
	local newRaidRoster = {};
	local newRaidRoster2 = {};
	
	-- Reset the Subgroups table
	for i = 1, NUM_RAID_GROUPS do
		BENECAST_RAID_SUBGROUPS[i] = 0;
	end
	
	-- Reset all the buttons
	BeneCast_ClearRaid();
	
	-- Set up all the buttons
	for i = 1, GetNumRaidMembers() do
		name, rank, subgroup, level, class, fileName, zone, online, isDead = GetRaidRosterInfo(i);
		if ( name ) then
			BENECAST_RAID_SUBGROUPS[subgroup] = BENECAST_RAID_SUBGROUPS[subgroup]+1;
			newRaidRoster['raid' .. i] = {
				name = name,
				rank = rank,
				subgroup = subgroup,
				level = level,
				class = class,
				fileName = fileName,
				zone = zone,
				online = online,
				isDead = isDead,
			};
			newRaidRoster2[name] = 'raid' .. i;
			local raidbutton = 'BeneCastRaidButton' .. ((subgroup-1)*MEMBERS_PER_RAID_GROUP + BENECAST_RAID_SUBGROUPS[subgroup]);
			local color_r = RAID_CLASS_COLORS[fileName].r;
			local color_g = RAID_CLASS_COLORS[fileName].g;
			local color_b = RAID_CLASS_COLORS[fileName].b;
			if ( BENECAST_RAID_LIST[name] ) then
				getglobal(raidbutton):SetChecked('true');
			end
			getglobal(raidbutton):Enable();
			getglobal(raidbutton .. 'Empty'):SetText(nil);
			getglobal(raidbutton .. 'Empty'):SetTextColor(GRAY_FONT_COLOR.r, GRAY_FONT_COLOR.g, GRAY_FONT_COLOR.b);
			getglobal(raidbutton .. 'Name'):SetText(name);
			getglobal(raidbutton .. 'Name'):SetTextColor(color_r, color_g, color_b);
			getglobal(raidbutton .. 'Level'):SetText(level);
			getglobal(raidbutton .. 'Level'):SetTextColor(color_r, color_g, color_b);
			getglobal(raidbutton .. 'Class'):SetText(class);
			getglobal(raidbutton .. 'Class'):SetTextColor(color_r, color_g, color_b);		
			
			if UnitExists('raidpet' .. i) then
				local punit = 'raidpet' .. i;
				newRaidRoster[punit] = {};
				local petname = UnitName(punit);
				newRaidRoster[punit].name = newRaidRoster['raid' .. i].name .. '|' .. petname;
				newRaidRoster2[newRaidRoster[punit].name] = punit;
				newRaidRoster[punit].rank = rank;
				newRaidRoster[punit].subgroup = subgroup;
				newRaidRoster[punit].level = UnitLevel(punit);
				newRaidRoster[punit].class = UnitClass(punit);
				newRaidRoster[punit].zone = zone;
				newRaidRoster[punit].online = online;
				newRaidRoster[punit].isDead = false;
			end
		end
	end
			
	-- Replace old raid-roster with new
	BENECAST_RAID_ROSTER = newRaidRoster;
	BENECAST_RAID_ROSTER2 = newRaidRoster2;
	
	-- Attach the Raid Panels to the correct frames
	BeneCast_AttachRaidPanels(BeneCastConfig.RaidSnapTo);
	
	-- Redraw the buttons for the raid
	for i = 6,45 do
		BeneCast_UpdateButtons(i);
	end
	for i = 51,90 do
		BeneCast_UpdateButtons(i);
	end

end

-- *****************************************************************************
-- BeneCast Unit Helper functions
-- *****************************************************************************

-- Function to determine if a unit is in your raid group
function BeneCast_UnitInRaid(unit)

	return UnitPlayerOrPetInRaid(unit);
	
end

-- Function to determine if a unitid is 'raid'..x
function BeneCast_UnitIsRaidID(unit)

	if string.find(unit,'raid') and not string.find(unit,'raidpet') then
		return true;
	end
	return nil;
	
end

-- Function to determine if a unitid is x..'pet'..y
function BeneCast_UnitIsPetID(unit)

	return string.find(unit, 'pet');
	
end

-- Function to determine if a unitid is 'raidpet'..x
function BeneCast_UnitIsRaidPetID(unit)

	return string.find(unit, 'raidpet');
	
end


-- Function to determine if a unit is a party pet
function BeneCast_UnitIsPartyPet(unit)

	return not UnitIsPlayer(unit) and UnitPlayerOrPetInParty(unit);
	
end

-- Function to determine if a unit is a raid pet
function BeneCast_UnitIsRaidPet(unit)

	return not UnitIsPlayer(unit) and UnitPlayerOrPetInRaid(unit);
	
end

-- Function to determine the BeneCastPanel number(s) of a UnitId
-- might return >1 number if it's a raidid
function BeneCast_UnitID(unit)

	if ( unit ) then
		if string.find(unit,'raid') then
			if BENECAST_RAID_PANELS3[unit] then
				local resultval = { };
				for i, parentframe in pairs (BENECAST_RAID_PANELS3[unit]) do
					table.insert(resultval,BENECAST_RAID_PANELS[parentframe]);
				end
				return resultval;
			end
		end
		for i in pairs (BC_targets) do
			if unit == BC_targets[i] then
				return i;
			end
		end
	end
	return nil;

end

-- *****************************************************************************
-- BeneCast Equipment Scanning functions
-- *****************************************************************************

function BeneCast_UpdateHealingBonuses()

	BC_plusheal = 0;
	if BonusScanner then
		if BonusScanner.active and ( BeneCastConfig.DmgBasedHeal or 
	                                     ( not BeneCastConfig.DmgBasedHeal and BeneCastConfig.ReverseHoTs ) ) then
			BC_plusheal = BonusScanner.bonuses['HEAL'];
			if BeneCastConfig.Debug then
				local ph;
				if BC_plusheal then
					ph = '+' .. BC_plusheal;
				else
					ph = 'nil';
				end
				PutDebugMsg('Plusheal is now ' .. ph);
			end
		end
	end
	--Call the next hook
	if BC_BonusScanner_HookFunction ~= nil then
		BC_BonusScanner_HookFunction();
	end

end

-- *****************************************************************************
-- BeneCast tooltip parsing functions
-- *****************************************************************************

-- Functions for parsing healing info from tooltip. Thanks Danboo!
local function parse_integer(text, index)

	if ( not text ) then
		return 0;
	end
	
	local i = 1;
	-- Replace ',' with '.', needed for German client. Thanks Auric!
	text = string.gsub(text, ',', '.');
	for integer in string.gmatch(text, '%d+%.?%d*') do
		if ( i == index ) then
			return tonumber(integer);
		end
		i = i + 1;
	end

end

-- Pull the first integer
local function pi_first(text)
	return parse_integer(text, 1);
end

-- Pull the second integer
local function pi_second(text)
	return parse_integer(text, 2);
end

-- Pull the second integer
local function pi_third(text)
	return parse_integer(text, 3);
end

-- Pull the second integer
local function pi_fourth(text)
	return parse_integer(text, 4);
end

-- Mapping of healing spell lines to heal amount parsers. Thanks Danboo!
-- Spell parsing is slightly different in the German client
local BC_spell_parsers = {}
if ( GetLocale() == 'deDE' ) then
	BC_spell_parsers = {
		[BENECAST_STRINGS.CLASS_SHAMAN] = {
				[BENECAST_STRINGS.HEAL_HEALING_WAVE] = pi_second,
				[BENECAST_STRINGS.HEAL_LESSER_HEALING_WAVE] = pi_second,
				[BENECAST_STRINGS.HEAL_CHAIN_HEAL] = pi_second,
		},
		[BENECAST_STRINGS.CLASS_PRIEST] = {
				[BENECAST_STRINGS.HEAL_LESSER_HEAL] = pi_second,
				[BENECAST_STRINGS.HEAL_HEAL] = pi_second,
				[BENECAST_STRINGS.HEAL_GREATER_HEAL] = pi_second,
				[BENECAST_STRINGS.HEAL_FLASH_HEAL] = pi_second,
				[BENECAST_STRINGS.HEAL_RENEW] = pi_second,
				[BENECAST_STRINGS.HEAL_PRAYER_OF_HEALING] = pi_second,
		},
		[BENECAST_STRINGS.CLASS_DRUID] = {
				[BENECAST_STRINGS.HEAL_HEALING_TOUCH] = pi_second,
				[BENECAST_STRINGS.HEAL_REJUVENATION] = pi_second,
				[BENECAST_STRINGS.HEAL_REGROWTH] = pi_second,
				[BENECAST_STRINGS.HEAL_TRANQUILITY] = pi_third,
		},
		[BENECAST_STRINGS.CLASS_PALADIN] = {
				[BENECAST_STRINGS.HEAL_HOLY_LIGHT] = pi_second,
				[BENECAST_STRINGS.HEAL_FLASH_OF_LIGHT] = pi_second,
				[BENECAST_STRINGS.HEAL_LAY_ON_HANDS] = pi_second,
				[BENECAST_STRINGS.HEAL_HOLY_SHOCK] = pi_fourth,
		},
		[BENECAST_STRINGS.CLASS_HUNTER] = {
				[BENECAST_STRINGS.HEAL_MEND_PET] = pi_first,
		},
		[BENECAST_STRINGS.CLASS_WARLOCK] = {
				[BENECAST_STRINGS.HEAL_HEALTH_FUNNEL] = pi_first,
		},
	};
else
	BC_spell_parsers = {
		[BENECAST_STRINGS.CLASS_SHAMAN] = {
			[BENECAST_STRINGS.HEAL_HEALING_WAVE] = pi_second,
			[BENECAST_STRINGS.HEAL_LESSER_HEALING_WAVE] = pi_second,
			[BENECAST_STRINGS.HEAL_CHAIN_HEAL] = pi_second,
		},
		[BENECAST_STRINGS.CLASS_PRIEST] = {
			[BENECAST_STRINGS.HEAL_LESSER_HEAL] = pi_second,
			[BENECAST_STRINGS.HEAL_HEAL] = pi_second,
			[BENECAST_STRINGS.HEAL_GREATER_HEAL] = pi_second,
			[BENECAST_STRINGS.HEAL_FLASH_HEAL] = pi_second,
			[BENECAST_STRINGS.HEAL_RENEW] = pi_first,
			[BENECAST_STRINGS.HEAL_PRAYER_OF_HEALING] = pi_second,
		},
		[BENECAST_STRINGS.CLASS_DRUID] = {
			[BENECAST_STRINGS.HEAL_HEALING_TOUCH] = pi_second,
			[BENECAST_STRINGS.HEAL_REJUVENATION] = pi_first,
			[BENECAST_STRINGS.HEAL_REGROWTH] = pi_second,
			[BENECAST_STRINGS.HEAL_TRANQUILITY] = pi_first,
		},
		[BENECAST_STRINGS.CLASS_PALADIN] = {
			[BENECAST_STRINGS.HEAL_HOLY_LIGHT] = pi_second,
			[BENECAST_STRINGS.HEAL_FLASH_OF_LIGHT] = pi_second,
			[BENECAST_STRINGS.HEAL_LAY_ON_HANDS] = pi_second,
			[BENECAST_STRINGS.HEAL_HOLY_SHOCK] = pi_fourth,
		},
		[BENECAST_STRINGS.CLASS_HUNTER] = {
				[BENECAST_STRINGS.HEAL_MEND_PET] = pi_first,
		},
		[BENECAST_STRINGS.CLASS_WARLOCK] = {
				[BENECAST_STRINGS.HEAL_HEALTH_FUNNEL] = pi_first,
		},
	};
end

-- *****************************************************************************
-- BeneCast Spell Loading function
-- *****************************************************************************

function BeneCast_SwitchPriestHeals()
 	if BeneCastConfig['DmgBasedHeal'] then
		BeneCast_SpellTypes[BENECAST_STRINGS.CLASS_PRIEST][BENECAST_STRINGS.HEAL_LESSER_HEAL] = 'efficient';
		BeneCast_SpellTypes[BENECAST_STRINGS.CLASS_PRIEST][BENECAST_STRINGS.HEAL_HEAL] = 'efficient';
		BeneCast_SpellTypes[BENECAST_STRINGS.CLASS_PRIEST][BENECAST_STRINGS.HEAL_GREATER_HEAL] = 'efficient';
		BeneCast_SpellTypes_Sorting[BENECAST_STRINGS.CLASS_PRIEST]['efficient'] =
			{ BENECAST_STRINGS.HEAL_LESSER_HEAL,
		          BENECAST_STRINGS.HEAL_HEAL,
		          BENECAST_STRINGS.HEAL_GREATER_HEAL };
	else
		BeneCast_SpellTypes[BENECAST_STRINGS.CLASS_PRIEST][BENECAST_STRINGS.HEAL_LESSER_HEAL] = 'efficient';
		BeneCast_SpellTypes[BENECAST_STRINGS.CLASS_PRIEST][BENECAST_STRINGS.HEAL_HEAL] = 'efficient2';
		BeneCast_SpellTypes[BENECAST_STRINGS.CLASS_PRIEST][BENECAST_STRINGS.HEAL_GREATER_HEAL] = 'efficient3';
		BeneCast_SpellTypes_Sorting[BENECAST_STRINGS.CLASS_PRIEST]['efficient'] =
			{ BENECAST_STRINGS.HEAL_LESSER_HEAL };
		BeneCast_SpellTypes_Sorting[BENECAST_STRINGS.CLASS_PRIEST]['efficient2'] =
			{ BENECAST_STRINGS.HEAL_HEAL };
		BeneCast_SpellTypes_Sorting[BENECAST_STRINGS.CLASS_PRIEST]['efficient3'] =
			{ BENECAST_STRINGS.HEAL_GREATER_HEAL };
	end
end

--Sort two spells by order in BeneCast_SpellTypes_Sorting
function BeneCast_GetSpellPos(spelltype, spellname)
	local pos = 0;
	for i, spellname2 in pairs (BeneCast_SpellTypes_Sorting[BC_class][spelltype]) do
		if spellname2 == spellname then
			pos = i;
			break;
		end
	end
	return pos;
end

function BeneCast_SortTwoSpells(a,b)
	if (a.name ~= b.name) then
		return BeneCast_GetSpellPos(a.spelltype,a.name) < BeneCast_GetSpellPos(b.spelltype,b.name);
	else
		return a.mana < b.mana
	end
end

-- Fill in the BC_spell_data table by iterating over spells in spellbook
-- and parsing tool tips for info. Thanks Danboo!
function BeneCast_LoadSpellData()
	PutDebugMsg('Parsing Spell Data from spellbook');
	
	BeneCast_SwitchPriestHeals();
	BC_spell_data = {}; 
	
	-- Iterator to move through the spellbook
	local i = 1;

	while true do
		
		-- Spell at position i in the player's spellbook
		local spell_name, spell_rank = GetSpellName(i, SpellBookFrame.bookType);
		
		-- If there is no spell at i, break out of the loop
		if not spell_name then
			do break end
		end
		
		-- Load up the spell_type based on the spell name and player class
		local spell_type = BeneCast_SpellTypes[BC_class][spell_name];

		-- If the spell_type exists do stuff
		if ( spell_type ) then
						
			-- Set BeneCast_Tooltip to the spell info
			-- Be sure to clear it first
			if not BeneCast_Tooltip:IsOwned(WorldFrame) then
				BeneCast_Tooltip:SetOwner(WorldFrame, 'ANCHOR_NONE');
			end
			BeneCast_Tooltip:ClearLines();
			BeneCast_Tooltip:SetSpell(i, SpellBookFrame.bookType);
			
			local mana_cost, heal_amount, dot_duration;
			local useTC = false;
			if not useTC then
				-- If this spell has a set mana cost set mana_cost to it
				if ( not mana_cost ) and ( BeneCast_TooltipTextLeft2:GetText() ) then
					if ( string.find(BeneCast_TooltipTextLeft2:GetText(), '^(%d+) Mana') ) then
						_,_,mana_cost = string.find(BeneCast_TooltipTextLeft2:GetText(), '^(%d+) Mana');
					end
				end			
			
				-- If the spell_type is a heal spell then set heal_amount to the amount this spell heals
				if ( string.find(spell_type,'efficient') or 
				     spell_type == 'emergency' or 
				     string.find(spell_type,'instant') or 
				     spell_type == 'group' or spell_type == 'group2' ) then
					if BC_spell_parsers[BC_class][spell_name] ~= nil then
						if ( BeneCast_TooltipTextLeft5:GetText() ) then
							heal_amount = BC_spell_parsers[BC_class][spell_name](BeneCast_TooltipTextLeft5:GetText());
						elseif ( BeneCast_TooltipTextLeft4:GetText() ) then
							heal_amount = BC_spell_parsers[BC_class][spell_name](BeneCast_TooltipTextLeft4:GetText());
						else
							heal_amount = BC_spell_parsers[BC_class][spell_name](BeneCast_TooltipTextLeft3:GetText());
						end
					end
				end
			end
			-- If this spell does not have a set mana cost, but does have a rank set mana_cost to the rank
			-- This is to ensure correct spell ordering
			if ( not mana_cost ) and ( spell_rank ~= '' ) then
				_, _, mana_cost = string.find(spell_rank, '%d+');
			end
			-- If this spell does not have a set mana cost or rank, set mana_cost to zero
			-- I'm guessing there is no need to order a spell type if it has no ranks
			if ( not mana_cost ) then
				mana_cost = 0;
			end

			-- If the player class is Paladin and the spell is Lay on Hands set heal_amount to zero
			-- This is to ensure that the highest rank is always used
			if BC_class == BENECAST_STRINGS.CLASS_PALADIN and spell_type == 'loh' then
				heal_amount = 0;
			end
			if not heal_amount then
				heal_amount = 0;
			end
							
			-- If there isn't anything in BC_spell_data[spell_type] yet then create a table for it
			if ( not BC_spell_data[spell_type] ) then
				BC_spell_data[spell_type] = {};
				BC_spell_data[spell_type]['spells'] = {};
			end
			
			local entry;
			-- If the entry type is a heal spell be sure to include rank and max_heal information
			if ( string.find(spell_type,'efficient') or 
			     spell_type == 'emergency' or 
			     string.find(spell_type,'instant') or 
			     spell_type == 'group' or spell_type == 'group2' ) then
				entry = { name = (spell_name),
					rank = (spell_rank),
					max_heal = tonumber(heal_amount),
					mana = tonumber(mana_cost),
					spelltype = spell_type, 
					duration = dot_duration,
					id = i };
			-- If the entry is a cure you don't need the rank
			elseif ( spelltype == 'poison' or spelltype == 'disease' or spelltype == 'magic' or spelltype == 'curse' ) then
				entry = {name = (spell_name),
					mana = tonumber(mana_cost),
					spelltype = spell_type, 
					duration = dot_duration,
					id = i };
			-- Otherwise it's a buff and include the spell rank if you can
			elseif ( spell_rank ~= '' ) then
				entry = {name = (spell_name),
					rank = (spell_rank),
					mana = tonumber(mana_cost),
					spelltype = spell_type,
					duration = dot_duration,
					id = i };
			-- If there is no rank, don't put a spell rank into the entry
			else
				entry = {name = (spell_name),
					mana = tonumber(mana_cost),
					spelltype = spell_type,
					duration = dot_duration,
					id = i };
			end
			
			-- Insert the entry into the appropriate table
			table.insert(BC_spell_data[spell_type]['spells'], entry);

		end

		i = i + 1;

	end
	
	-- Sort the spells by mana cost within each spell type
	for type in pairs (BC_spell_data) do
		table.sort(BC_spell_data[type]['spells'], BeneCast_SortTwoSpells);
	end
	
	-- If the spell_type does not have a texture associated with it set it
	for type in pairs (BC_spell_data) do
		local texture;
		-- Make sure the texture is for the last entry in the table
		local i = table.getn(BC_spell_data[type]['spells']);
		BC_spell_data[type].spellcount = i;
		texture = GetSpellTexture(BC_spell_data[type]['spells'][i].id, SpellBookFrame.bookType);
		BC_spell_data[type].texture = texture;
		BC_spell_data[type].type = type;
	end
	
end


-- *****************************************************************************
-- BeneCast spell choose functions
-- *****************************************************************************

-- Interpolates the health of the given unit (used for those outside the party). Thanks Danboo!
-- Returns an estimated health for a target outside the party
function BeneCast_UnitInterpolateHealth(unit, health_percent)
	
	local class      = UnitClass(unit);
	if class == nil then
		class = BENECAST_STRINGS.CLASS_WARRIOR;
	end
	local level      = UnitLevel(unit);
	local slope      = ( BC_class_health[class]['60'] - BC_class_health[class]['1'] ) / 59;
	local health_max = level * slope + BC_class_health[class]['1'];
	local health     = health_max * health_percent / 100;
	return floor(health), floor(health_max);
	
end

-- *****************************************************************************
-- BeneCast Slash Command Handler
-- *****************************************************************************

-- Slash command handler
-- The only slash command is to show the config GUI
function BeneCast_SlashCommandHandler(msg)

	if string.find(msg,'debug') then
		BeneCastConfig['Debug'] = not BeneCastConfig['Debug'];
		PutMsg('command: ' .. msg .. ' , debug is now '..(BeneCastConfig['Debug'] and 'on' or 'off'));
	elseif string.find(msg,'peekb (.+)') then
		local varname;
		_, _, varname = string.find(msg,'peekb (.+)');
		if varname then
			PutMsg('command: ' .. msg .. ' , BeneCastConfig['..varname..'] = ' .. ( (not BeneCastConfig[varname]) and 'nil' or BeneCastConfig[varname] ));
		else
			PutMsg('command: ' .. msg .. ' , invalid parameter');
		end
	elseif string.find(msg,'peek (.+) (.+)') then
		local varname, varname2;
		_, _, varname, varname2 = string.find(msg,'peek (.+) (.+)');
		if varname and varname2 then
			PutMsg('command: '..msg..' , BeneCastConfig['..varname..'][' .. varname2 .. '] = '..( (not BeneCastConfig[varname][varname2]) and 'nil' or BeneCastConfig[varname] ) );
		else
			PutMsg('command: ' .. msg .. ' , invalid parameter');
		end
	elseif string.find(msg,'peek (.+)') then
		local varname;
		_, _, varname = string.find(msg,'peek (.+)');
		if varname then
			if BeneCastConfig[varname] then
				PutMsg('command : ' .. msg .. ' , BeneCastConfig[' .. varname ..
				  '] = ' .. BeneCastConfig[varname]);
			else
				PutMsg('command : ' .. msg .. ' , BeneCastConfig[' .. varname .. 
				  '] = nil or not found');
			end
		else
			PutMsg('command : ' .. msg .. ' , invalid parameter');
		end
	elseif string.find(msg,'snapto2 (.+)') then
		local addonpar;
		_, _, addonpar = string.find(msg,'snapto2 (.+)');
		if addonpar then
			PutMsg('trying to snapto ' .. addonpar);
			SetBeneCastSnapTo2(addonpar);
		else
			PutMsg('snapto2 had no value after it');
		end
	elseif string.find(msg,'snapto (.+)') then
		local addonname = nil;
		_, _, addonname = string.find(msg,'snapto (.+)');
		if addonname then
			PutMsg('trying to snapto ' .. addonname);
			SetBeneCastSnapTo(addonname);
		else
			PutMsg('snapto had no addonname after it');
		end
	elseif string.find(msg,'firstaid (.+)') then
		local item;
		_, _, item = string.find(msg,'firstaid (.+)');
		if item then
			BC_FirstAidItem = item;
			PutMsg('setting First Aid item to ' .. BC_FirstAidItem);
			for i,attached in pairs (BC_attached_targets) do
				if attached then
					BeneCast_UpdateButtons(i);
				else
					PutDebugMsg('Not updating "'..BC_targets[i]..'" because it was not previously attached.');
				end
			end
		else
			PutMsg('firstaid had no value after it');
		end
	else
		BeneCastOptionFrameToggle();
	end
	
end

-- *****************************************************************************
-- BeneCast Key Binding functions
-- *****************************************************************************

-- Function to toggle showing of all buttons
function BeneCast_ToggleBuffs()
	
	-- Toggle showallbuttons
	BeneCastConfig.ShowAllButtons = not BeneCastConfig.ShowAllButtons;
	
	for i in pairs (BC_targets) do
		BeneCast_UpdateButtons(i);
	end

end

-- *****************************************************************************
-- BeneCast Party Notification functions
-- *****************************************************************************

-- Function for Party Notification
-- Modified SpeakSpell by Danboo. Thanks Danboo!
-- Returns true if a notification message was sent
function BeneCast_PartyNotify(target, spell, ismax, duration)

	if ( not BeneCastConfig.Notification ) then
		return nil;
	end
	
	-- If there is no duration, give a duration of 0
	if ( not duration ) then
		duration = 0;
	end
	
	-- Don't show time if notifytime is nil or duration is 0
	local time = nil;
	if ( duration > 0 and BeneCastConfig.NotifyTime ) then
		time = true;
	end
	
	-- Don't show rank if spell.rank is nil
	local rank = nil;
	if ( spell.rank and spell.rank ~= 'Rank 0' ) then
		rank = true;
	else
		spell.rank = 'Rank 0';
	end
	
	-- Figure out the spelltype for display purposes
	local type = BeneCast_SpellTypes[BC_class][spell.name];
	
	if type == nil then 
		PutDebugMsg('Unknown spell class ' .. BC_class .. ', spell ' .. spell.name);
		return nil;
	end
	
	local healamount_text = '';
	
	-- If the type is a heal prepare the healamount_text
	if ( string.find(type,'efficient') or type == 'emergency' or string.find(type,'instant') or 
	     type == 'group' or type == 'group2' ) then
		healamount_text = ' (';
		if spell.max_heal then
			healamount_text = healamount_text .. spell.max_heal;
		else
			healamount_text = healamount_text .. '???';
		end
		if BC_plusheal then
			if BC_plusheal ~= 0 then
				healamount_text = healamount_text .. '+' .. BC_plusheal .. ')';
			else
				healamount_text = healamount_text .. ')';
			end
		else
			healamount_text = healamount_text .. ')';
		end
	end

	duration = string.format('%.1f', duration / 1000);

	local message_text;
	
	-- Set the message text based on what options are checked
	if ( rank and BeneCastConfig.NotifyRank and time ) then
		message_text = BENECAST_STRINGS.NOTIFY_ALL;
	elseif ( rank and BeneCastConfig.NotifyRank ) then
		message_text = BENECAST_STRINGS.NOTIFY_RANK;
	elseif ( time ) then
		message_text = BENECAST_STRINGS.NOTIFY_TIME;
	else
		message_text = BENECAST_STRINGS.NOTIFY;
	end
	
	message_text = string.gsub(message_text, '%%s', spell.name);
	message_text = string.gsub(message_text, '%%d', duration);
	message_text = string.gsub(message_text, '%%r', spell.rank);
	message_text = message_text .. healamount_text; --WINTROW.6
	
	-- First determine what type of spell it is
	local heal, cure, buff, res;
	if ( string.find(type,'efficient') or type == 'emergency' or 
	     string.find(type,'instant') or type == 'group' or type == 'group2' ) then
		heal = true;
	elseif ( type == 'poison' or type == 'disease' or type == 'disease2' or 
	         type == 'magic' or type == 'curse' ) then
		cure = true;
	elseif ( type == 'res' ) then
		res = true;
	else
		buff = true;
	end
	local rare;
	if type == 'res' or 
	   ( BC_class == BENECAST_STRINGS.CLASS_DRUID and type == 'buff3' ) or
	   ( BC_class == BENECAST_STRINGS.CLASS_HUNTER and type == 'selfbuff8' ) or
	   ( BC_class == BENECAST_STRINGS.CLASS_ROGUE and type == 'selfbuff5' ) or
	   ( BC_class == BENECAST_STRINGS.CLASS_WARRIOR and ( type == 'selfbuff1' or type == 'selfbuff4') ) or
	   ( BC_class == BENECAST_STRINGS.CLASS_PALADIN and ( type == 'partybuff3' or type == 'loh' ) ) then
		rare = true;
	end

	local channels_sent = {};
	
	for i, values in pairs (BeneCastOptionDropDowns) do
		local enable = values.enable;
		local ok = false;
		local cvar = values.cvar;
		local channel = BeneCastConfig[cvar];
		-- Make sure the same channel only gets 1 msg
		if not channels_sent[channel] then
			-- Check if the notification is enabled
			if BeneCastConfig[enable] then
				-- And the type of spell corresponds with the type of notification
				if ( enable == 'SelfCasts' ) and ( target == 'player' ) then
					ok = true;
				elseif ( enable == 'NotifyHeal' ) and heal then
					ok = true;
				elseif ( enable == 'NotifyCure') and buff then
					ok = true;
				elseif ( enable == 'NotifyRare' ) and rare then
					ok = true;
				elseif ( enable == 'NotifyRes' ) and res then
					ok = true;
				elseif ( enable == 'NotifyBuff' ) and buff then
					ok = true;
				end
			end
		end
		if ok then
			local msg_text = message_text;
		
			if channel == 'WHISPER' then
				-- If notifying the target, address them as 'you'
				msg_text = string.gsub(msg_text, '%%t', BENECAST_STRINGS.NOTIFY_YOU);
			-- In laggy environments if the player's target was deselected while casting it does not have a name
			elseif ( UnitExists(target) ) then
				local name = UnitName(target);
				msg_text = string.gsub(msg_text, '%%t', name);
			else
				msg_text = string.gsub(msg_text, '%%t', BENECAST_STRINGS.NOTIFY_UNKNOWN_ENTITY);
			end
		
			-- Send the message on the appropriate channel
			if ( channel == 'SELF' ) then
				PutMsg(msg_text);
			elseif ( channel == 'PARTY' and UnitExists('party1') ) then
				SendChatMessage(msg_text, 'PARTY');
			elseif ( channel == 'SAY' ) then
				SendChatMessage(msg_text, 'SAY');
			elseif ( channel == 'RAID' and UnitExists('raid1') ) then
				SendChatMessage(msg_text, 'RAID');
			elseif ( channel == 'WHISPER' and target ~= 'player' and UnitIsPlayer(target) ) then
				local name = UnitName(target);
				SendChatMessage(msg_text, 'WHISPER', nil, name);
			elseif ( channel == 'USER' ) then
				SendChatMessage(msg_text, 'CHANNEL', nil, GetChannelName(BeneCastConfig.UserChannel));
			-- Thnx to Shimavak for the emote-code
			elseif ( channel == 'EMOTE' ) then 
				SendChatMessage(BENECAST_STRINGS.NOTIFY_EMOTE_PRE .. string.sub(msg_text,2), 'EMOTE');
			end
				
			-- Make sure the same channel doesn't get >1 msg
			channels_sent[channel] = true;
		end
	end
	
	-- Message sent, return true
	return true;

end

-- Function for Party Notification on failures and interruptions
-- Modified SpeakSpell by Danboo. Thanks Danboo!
function BeneCast_PartyNotifyFail(event)

	local message_text;
	
	if BeneCastConfig.NotifyFailure then

		-- Set the message text based on what event called PartyNotifyFail
		-- SPELLCAST_CANCELED is really SPELLCAST_INTERRUPTED if the interruption was caused by the player
		-- To know if interruptions are caused by the player, SPELLCAST_STOP is called just before SPELLCAST_INTERRUPTED
		if event == 'SPELLCAST_FAILED' then
			message_text = BENECAST_STRINGS.NOTIFY_FAILED;
		elseif event == 'SPELLCAST_INTERRUPTED' then
			message_text = BENECAST_STRINGS.NOTIFY_INTERRUPTED;
		elseif event == 'SPELLCAST_CANCELED' then
			message_text = BENECAST_STRINGS.NOTIFY_CANCELED;
		end
		
		local channel = BeneCastConfig.NotifyFailureChannel;
	
		-- Send the message on the appropriate channel
		if ( channel == 'SELF' ) then
			PutMsg(msg_text);
		elseif ( channel == 'PARTY' and UnitExists('party1') ) then
			SendChatMessage(msg_text, 'PARTY');
		elseif ( channel == 'SAY' ) then
			SendChatMessage(msg_text, 'SAY');
		elseif ( channel == 'RAID' and UnitExists('raid1') ) then
			SendChatMessage(msg_text, 'RAID');
		elseif ( channel == 'WHISPER' and target ~= 'player' and UnitIsPlayer(target) ) then
			local name = UnitName(target);
			SendChatMessage(msg_text, 'WHISPER', nil, name);
		elseif ( channel == 'USER' ) then
			SendChatMessage(msg_text, 'CHANNEL', nil, GetChannelName(BeneCastConfig.UserChannel));
		-- Thnx to Shimavak for the emote-code
		elseif ( channel == 'EMOTE' ) then 
			SendChatMessage(message_text, 'EMOTE');
		end
		
	end

end

-- *****************************************************************************
-- BeneCastButton functions
-- *****************************************************************************

function BeneCast_GetUnitIDFromPanelID(id)
	local target = nil;
	if ( id >= 6 and id <= 45 ) or ( id >= 51 and id <= 90 ) then
		if BENECAST_RAID_PANELS2[id] then
			target = BENECAST_RAID_PANELS2[id]['unitid'];
		end
	else
		target = BC_targets[id];
	end
	return target;
end

-- Make a tooltip for the BeneCastButton based on the options set
function BeneCastButton_OnEnter()

	if ( this.spelltype == 'assist' ) then
		GameTooltip:SetOwner(this, 'ANCHOR_BOTTOMRIGHT');
		GameTooltip:SetText(BENECAST_STRINGS.TEXT_ASSIST);
		return;
	end
	if ( BeneCastConfig.ShowTooltips and not BeneCastConfig.ShowTooltipsName  ) then
		if this.spellid then
			GameTooltip:SetOwner(this, 'ANCHOR_BOTTOMRIGHT');
			GameTooltip:SetSpell(this.spellid, SpellBookFrame.bookType);
		end
	end
	if ( BeneCastConfig.ShowTooltips and BeneCastConfig.ShowTooltipsName ) then
		GameTooltip:SetOwner(this, 'ANCHOR_BOTTOMRIGHT');
		GameTooltip:SetText(this.spellname);
	end
	if this.reason then
		if this.reason ~= '' then
			PutDebugMsg('Fade-reason: ' .. this.reason);
		end
	end

end

-- Update the cooldown for a button on events
function BeneCastButton_UpdateCooldown()

	-- If the unit for this button does not exist return
	if ( not UnitExists(BC_targets[this:GetParent():GetID()]) ) then
		this:GetParent():Hide();
		return;
	end
	
	-- No need to update for a hidden button
	if ( not this:IsVisible() ) then
		return;
	end
	
	local spell;
	-- Figure out this button's spelltype
	local spelltype = this.spelltype;
	-- If there is no spell associated with the button
	if not spelltype or this.spellid == nil or this.spellid == 0 then
		return;
	end
	
	-- Cooldown stuff that I'm not quite sure how exactly it all works
	local cooldown = getglobal(this:GetName()..'Cooldown');
	local iconTexture = getglobal(this:GetName()..'Icon');
	local start, duration, enable = GetSpellCooldown(this.spellid, SpellBookFrame.bookType);
	CooldownFrame_SetTimer(cooldown, start, duration, enable);
	if ( enable == 1 ) then
		iconTexture:SetVertexColor(1.0, 1.0, 1.0);
	else
		iconTexture:SetVertexColor(0.4, 0.4, 0.4);
	end

end

-- Register events for BeneCastButtons
function BeneCastButton_OnLoad()

	this:RegisterEvent('SPELL_UPDATE_COOLDOWN');

end

-- *****************************************************************************
-- BeneCast Button Drawing/Updating Functions
-- *****************************************************************************

-- Function used to show a BeneCast button
-- Parameters spell_type, and the first free button available
-- 'Free' buttons are buttons that are not being used
-- Returns true if a button was created
function BeneCast_ShowButton(type, freebutton, id, member) --WINTROW.6 'member' is a parameter now

	if InCombatLockdown() then
		return;
	end
	local made_button = nil;
	-- Figure out who this button belongs too
	--WINTROW.6 local member = BC_targets[id]; --Is parameter now
	local class = UnitClass(member);
	if not class then
		class = BENECAST_STRINGS.CLASS_WARRIOR;
	end
	local classnum = BC_classes[class];
	if classnum == nil then
		PutMsg('Cant find classnum for class ' .. class);
	end
	
	--Only if a hunter/warlock is targetting his pet show the petheal
	if ( BC_class == BENECAST_STRINGS.CLASS_WARLOCK or BC_class == BENECAST_STRINGS.CLASS_HUNTER ) then
		if ( ( member ~= 'player' and not ( member == 'target' and UnitIsUnit('pet','target') ) ) and type == 'efficient' ) then
			return nil;
		end
	end
	
	-- If the player has the spell, wants it shown, and freebutton is <= BeneCastConfig.ButtonNumber then continue
	if ( BC_spell_data[type] and 
	     ( ( member ~= 'player' and BeneCastConfig[classnum][type] ) or
	       ( member == 'player' and BeneCastConfig[1][type] ) ) and 
	     freebutton <= BeneCastConfig.ButtonNumber
	   ) then
		local button = getglobal('BeneCastPanel' .. id .. 'Button' .. freebutton);
		-- Set made_button to true
		made_button = true;
		-- Set up the texture for the button
		local icon = getglobal('BeneCastPanel' .. id .. 'Button' .. freebutton .. 'Icon');
		icon:SetTexture(BC_spell_data[type].texture);
		button.spelltype = type;
		if type == 'assist' then
			button:SetAttribute('type','target');
			button:SetAttribute('unit',member .. 'target');
			button.spellid = nil;
			button.spellname = BENECAST_STRINGS.TEXT_ASSIST;
		elseif type == 'firstaid' then
			button:SetAttribute('type','macro');
			button:SetAttribute('macrotext','/use ' .. BC_FirstAidItem);
			button:SetAttribute('unit',member);
			local i = BC_spell_data[type].spellcount;
			button.spellid = BC_spell_data[type]['spells'][i].id;
			button.spellname = BENECAST_STRINGS.FIRST_AID;
		else
			button:SetAttribute('type','spell');
			local i = BC_spell_data[type].spellcount;
			local spellname = BC_spell_data[type]['spells'][i].name;
			if BC_spell_data[type]['spells'][i].rank then
				spellname = spellname .. '(' .. BC_spell_data[type]['spells'][i].rank .. ')';
			else
				spellname = spellname .. '()';
			end
			button.spellid = BC_spell_data[type]['spells'][i].id;
			button.spellname = spellname;
			button:SetAttribute('spell',spellname);
			button:SetAttribute('unit',member);
		end
		button:Show();
		button:SetAlpha(1); --reset alpha
		if not BC_buttons_by_spell[member] then
			BC_buttons_by_spell[member] = {};
		end
		BC_buttons_by_spell[member][type] = button:GetName();
	end

	return made_button;
	
end

-- Function to fade buttons if they're not usable
function BeneCast_SetButtonFade(buttonname, showbutton, reason)
	if buttonname == nil then
		return;
	end
	local button = getglobal(buttonname);
	if button == nil then
		return;
	end
	button.reason = reason;
	local fadeicon = getglobal(buttonname .. 'Fade');
	if showbutton then
		fadeicon:Hide();
	else
		fadeicon:Show();
	end
end

-- Function to hide temporary buttons if they're not to be shown
function BeneCast_HideButton(buttonnum, id, member)

	local button = getglobal('BeneCastPanel' .. id .. 'Button' .. buttonnum);
	if InCombatLockdown() then
		return;
	end
	local fadeicon = getglobal('BeneCastPanel' .. id .. 'Button' .. buttonnum .. 'Fade');
	if ( fadeicon:IsShown() ) then
		fadeicon:Hide();
	end
	local rangeicon = getglobal('BeneCastPanel' .. id .. 'Button' .. buttonnum .. 'Range');
	if ( rangeicon:IsShown() ) then
		rangeicon:Hide();
	end
	if ( button:IsShown() ) then
		button:Hide();
	end
	if button.spelltype then
		if BC_buttons_by_spell[member] then
			BC_buttons_by_spell[member][button.spelltype] = nil;
		end
	end

end

--WINTROW.6 START
-- Function used to update buttons for a unitid
function BeneCast_UpdateButtonsForUnit(unitid)
	local panelids;
	panelids = BeneCast_UnitID(unitid);
	if panelids then
		if type(panelids) == 'table' then
			for i, id in pairs (panelids) do
				BeneCast_UpdateButtons(id);
			end
		else
			BeneCast_UpdateButtons(panelids);
		end
	end
end
--WINTROW.6 STOP

-- Function used to update buttons for all possible targets
function BeneCast_UpdateButtons(id)

	-- Sometimes the id is nil
	if not id then
		PutDebugMsg('BeneCast_UpdateButtons(): parameter id is nil');
		return;
	end
	
	local wasAttached = BC_attached_targets[id];
	BC_attached_targets[id] = false;
	
	if getglobal('BeneCastPanel' .. id) == nil then
		PutDebugMsg('Panel BeneCastPanel' .. id .. ' not found');
		return;
	end
	
	local panel = getglobal('BeneCastPanel' .. id);
	
	if ( id == 92 and not BeneCastConfig.ShowTargetTarget ) then
		if panel:CanChangeProtectedState() then
			panel:Hide();
		end
		return;
	end
	
	local disableall = false;
	local buttonstochange = {};
	
	-- Figure out who this frame belongs too
	local member = BeneCast_GetUnitIDFromPanelID(id);
	if member == nil then
		if panel:CanChangeProtectedState() then
			panel:Hide();
		end
		return;
	end
		
	-- If the player is dead then hide this panel and return unless this is a hunter Feigning
	--	or if we were showing this panel before, we want to reset that status.
	--if ( ( UnitIsDead('player') and not UnitBuff(member, 1) ) or UnitIsGhost('player') ) then
	--	getglobal('BeneCastPanel' .. id):Hide();
	--	BC_attached_targets[id] = wasAttached;
	--	return;
	--end

	-- Hide this BeneCastPanel frame and return if this member is not valid
	if not UnitExists(member) then
		if panel:CanChangeProtectedState() and not BC_ConsiderInCombat then
			panel:Hide();
		end
		return;
	end
	
	if ( UnitIsGhost('player') or not UnitIsConnected(member) ) then
		disableall = true;
	end

	-- Hide this BeneCastPanel frame if member is not friendly/mindcontrolled
	local curemagic = false;
	if UnitIsCharmed(member) then
		curemagic = true;
	end
	
	-- Disable this BeneCastPanel frame if member is not friendly
	local unitisfriend = true;
	if ( not UnitIsFriend('player', member) and not curemagic ) then
		if panel:CanChangeProtectedState() and not BC_ConsiderInCombat then
			panel:Hide();
			return;
		else
			disableall = true;
		end
		unitisfriend = false;
	end
	
	-- Hide this BeneCastPanel frame and return if this member is a raidid that is not being shown
	local name = UnitName(member);
	local raidenabled = BENECAST_RAID_LIST[name];
	if unitisfriend then
		if ( BeneCast_UnitIsRaidID(member) and ( not raidenabled or not BeneCastConfig.ShowRaid ) ) then
			if panel:CanChangeProtectedState() and not BC_ConsiderInCombat then
				panel:Hide();
				return;
			else
				disableall = true;
			end
		end
	end
	
	-- Hide this BeneCastPanel frame and return if this member is a pet and ShowPets is not enabled
	-- Except if the player is directly targetting the pet
	if ( member ~= 'target' ) and unitisfriend then
		if ( ( BeneCast_UnitIsPetID(member) or BeneCast_UnitIsPartyPet(member) ) and not BeneCastConfig.ShowPets ) then
			if panel:CanChangeProtectedState() and not BC_ConsiderInCombat then
				panel:Hide();
				return;
			else
				disableall = true;
			end
		end
	end
	
	-- If the player can ressurect this member then show the res button and return
	-- 	If there is a buff showing (UnitBuff()) then the player _cannot_ be dead, or they
	-- 	are a priest in Spirit of Redemption form, and so they shouldn't be ressurected.
	-- 	DEAD = No buffs.  Hunters do not need a special case here.
	-- 	We don't bother with ghosts, because we can't cast it on their name, we have to find
	-- 	their body.
	local showres = false;
	if unitisfriend then
		if ( UnitIsDead(member) and not UnitBuff(member, 1) ) then
			showres = true;
		end
	end
	
	local activeshapeshiftid = 0;
	local activeshapeshiftname = 'Caster';
	-- If player is a Druid and is shapeshifted hide all buttons
	if ( BC_class == BENECAST_STRINGS.CLASS_DRUID ) then
		-- Check all shapeshift forms to see if they're active
		local shapeshiftnum = GetNumShapeshiftForms();
		local icon, name, active;
		for i = 1,shapeshiftnum do
			icon, name, active = GetShapeshiftFormInfo(i);
			if ( active ) then
				activeshapeshiftid = i;
				activeshapeshiftname = name;
				do break end
			end
		end
	end
	
	-- Otherwise show the buttons. Show the BeneCastPanel frame first
	if panel:CanChangeProtectedState() then
		panel:Show();
	end
	
	-- If there is no class, set it to warrior
	local class = UnitClass(member);
	if not class then
		class = BENECAST_STRINGS.CLASS_WARRIOR;
	end
	
	for i, spelltype in pairs(BC_spellsort) do
		buttonstochange[spelltype] = {};
	end
	buttonstochange['assist'] = {};

	-- Show assist button
	buttonstochange['assist'].add = true;
	buttonstochange['assist'].show = true;
	
	-- Show Resurrection button
	buttonstochange['res'].add = true;
	buttonstochange['res'].show = showres;
	if not showres then
		buttonstochange['res'].reason = 'Unit not ressurectable';
	end
	
	-- Show First Aid
	buttonstochange['firstaid'].add = true;
	buttonstochange['firstaid'].show = (BC_FirstAidItem ~= nil);
	
	-- Show Heal buttons as persistant buttons
	buttonstochange['efficient'].add = true;
	buttonstochange['efficient2'].add = true;
	buttonstochange['efficient3'].add = true;
	buttonstochange['emergency'].add = true;
	buttonstochange['instant'].add = true;
	buttonstochange['efficient'].show = (activeshapeshiftid == 0);
	buttonstochange['efficient2'].show = (activeshapeshiftid == 0);
	buttonstochange['efficient3'].show = (activeshapeshiftid == 0);
	buttonstochange['emergency'].show = (activeshapeshiftid == 0) or (activeshapeshiftid == 6);
	buttonstochange['instant'].show = (activeshapeshiftid == 0) or (activeshapeshiftid == 6);
	if (activeshapeshiftid ~= 0) then
		buttonstochange['efficient'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
		buttonstochange['efficient2'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
		buttonstochange['efficient3'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
	end
	if (activeshapeshiftid ~= 0) and (activeshapeshiftid ~= 6) then
		buttonstochange['emergency'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
		buttonstochange['instant'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
	end
	
	-- Paladin's Lay on Hands spell
	buttonstochange['loh'].add = true;
	buttonstochange['loh'].show = true;

	-- Group heals are handled differently than other heals
	-- Only on the Shaman Group heal does targeting matter
	-- Show up only on player unless class is Shaman
	buttonstochange['group'].add = 
		( BC_class == BENECAST_STRINGS.CLASS_SHAMAN or member == 'player' );
	buttonstochange['group2'].add = ( member == 'player' );
	buttonstochange['group'].show = (activeshapeshiftid == 0) or (activeshapeshiftid == 6);
	buttonstochange['group2'].show = (activeshapeshiftid == 0);
	if (activeshapeshiftid ~= 0) and (activeshapeshiftid ~= 6) then
		buttonstochange['group'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
	end
	if activeshapeshiftid ~= 0 then
		buttonstochange['group2'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
	end
	
	-- Check to see if the member can be cured of a harmful effect
	-- Do this before showing buffs so Weakened Soul is detected before drawing buffs
	-- Iterator for debuffs
	local i = 1;
	
	-- Variables to determine if cures should be shown
	-- This is done in case there are multiples of one kind of effect
	local poisonshow = nil;
	local diseaseshow = nil;
	local magicshow = nil;
	local curseshow = nil;
	
	local weakenedsoul = nil;
	BC_spirit_of_redemption = nil;
	
	-- Cycle through the Units Debuffs to find removable effects
	if not BeneCast_Tooltip:IsOwned(WorldFrame) then
		BeneCast_Tooltip:SetOwner(WorldFrame, 'ANCHOR_NONE');
	end

	local debuffname, dispeltype;	
	while ( UnitDebuff(member, i) ) do

		debuffname, _, _, _, dispeltype = UnitDebuff(member, i);
		if ( debuffname ) then
			-- Check to see if this debuff is Weakened Soul and save it for when you get to the buffs
			if ( debuffname == BENECAST_STRINGS.AILMENT_WEAKENED_SOUL ) then
				weakenedsoul = true;
			end
			if ( debuffname == BENECAST_STRINGS.AILMENT_SPIRIT_OF_REDEMPTION ) then
				BC_spirit_of_redemption = true;
			end
	
			if dispeltype == BENECAST_STRINGS.AILMENT_POISON then
				poisonshow = true;
			elseif ( dispeltype == BENECAST_STRINGS.AILMENT_DISEASE ) then
				-- Paladins can cure diseases but don't have a spell with the 'disease' spelltype
				-- The Paladin spell that cures poisons also cures diseases
				if ( BC_class == BENECAST_STRINGS.CLASS_PALADIN ) then
					poisonshow = true;
				else
					diseaseshow = true;
				end
			elseif ( dispeltype == BENECAST_STRINGS.AILMENT_MAGIC ) then
				-- Paladins can cure magic with Cleanse which is of the 'poison' spelltype
				-- If the Paladin has Cleanse have it show Cleanse
				if ( BC_class == BENECAST_STRINGS.CLASS_PALADIN and BC_spell_data['poison'] ) then
					if BC_spell_data['poison']['spells'][2] then
						poisonshow = true;
					else
						magicshow = true;
					end
				else
					magicshow = true;
				end
			elseif ( dispeltype == BENECAST_STRINGS.AILMENT_CURSE ) then
				curseshow = true;
			end
		end
		i = i + 1;
	end
	
	-- Show Buff buttons as temporary buttons
	
	-- See if the groupbuff is on the intended target
	local groupbuffsshown = {};
	local groupblessingsshown = {};
	
	-- See if a mage's ward is on the player
	local wardshown = nil;
			
	-- Check to see if this member does not have castable buffs
	
	-- Reset the iterator for buffs
	i = 1;
	
	-- Clear all entries in BC_buffs_in_effect
	for j in pairs (BC_buffs_in_effect) do
		BC_buffs_in_effect[j].fade = false;
	end
	
	if UnitIsUnit(member,'player') then
		-- Set clearcasting state to nil
		BC_clearcasting = nil;
		local old_innerfocus = BC_innerfocus; --WINTROW.6
		BC_innerfocus = nil; --WINTROW.6
	end
	
	-- Cycle through the Units Buffs to find if the effect is on the unit
	local buffname = nil;
	while ( UnitBuff(member, i) ) do
				
		buffname = UnitBuff(member, i);
		if ( buffname ) then
			-- Only check these buffs if we're checking out the player ! (Thnx Shimavak)
			if UnitIsUnit(member,'player') then
				-- If the buff is Clearcasting then set clearcasting state to true 
				if ( buffname == BENECAST_STRINGS.BUFF_CLEARCASTING ) then
					BC_clearcasting = true;
				end
				if ( buffname == BENECAST_STRINGS.SELFBUFF_INNER_FOCUS ) then
					BC_innerfocus = true;
				end
				if ( buffname == BENECAST_STRINGS.AILMENT_SPIRIT_OF_REDEMPTION ) then
					BC_spirit_of_redemption = true;
				end
			end
			-- Now, if the buff is one of the uninteractable ones, we'll turn off the whole set of buttons				
			if ( buffname == BENECAST_STRINGS.BUFF_PHASE_SHIFT or buffname == BENECAST_STRINGS.PARTYBUFF_DIVINE_INTERVENTION ) then
				disableall = true;
			end
			
			local effect = BeneCast_SpellTypes[BC_class][buffname];
			-- If the effect is a groupbuff set showgroupbuff to true
			if ( effect ) then
				if ( ( BC_class == BENECAST_STRINGS.CLASS_PRIEST or 
				       BC_class == BENECAST_STRINGS.CLASS_DRUID or 
				       BC_class == BENECAST_STRINGS.CLASS_MAGE ) and string.find(effect,'groupbuff(.+)') ) then
				        local tmp;
				        _, _, tmp = string.find(effect,'groupbuff(.+)');
					table.insert(groupbuffsshown,tmp);
				elseif ( BC_class == BENECAST_STRINGS.CLASS_PALADIN and string.find(effect,'buffparty(.+)g') ) then
					local tmp;
					_, _, tmp = string.find(effect,'buffparty(.+)g');
					table.insert(groupblessingsshown,tmp);
				-- If the effect is a mage ward set wardshown to true
				elseif ( BC_class == BENECAST_STRINGS.CLASS_MAGE and ( effect == 'selfbuff12' or effect == 'selfbuff13' ) ) then
					wardshown = true;
				end 

				if ( BC_buffs_in_effect_by_name[effect] ) then
					local j = BC_buffs_in_effect_by_name[effect];
					-- Set fade for the type to true
					BC_buffs_in_effect[j].fade = true;
				else
					PutDebugMsg('Buff ' .. effect .. ' not found in BC_buffs_in_effect_by_name-table');
				end
			end
		else
			PutDebugMsg('buffname was false!');
		end
		i = i + 1;
		
	end
	
	--WINTROW.6 START
	if BeneCastConfig.Debug then
		if BC_innerfocus and not old_innerfocus then
			PutDebugMsg('Detected Inner Focus');
		end
	end
	--WINTROW.6 STOP
	
	-- Make _buff_shown for PW:F, AI, or MotW on the intended target true if they have the groupbuff on them
	if ( ( BC_class == BENECAST_STRINGS.CLASS_PRIEST or 
	       BC_class == BENECAST_STRINGS.CLASS_DRUID or 
	       BC_class == BENECAST_STRINGS.CLASS_MAGE ) and table.getn(groupbuffsshown) > 0 ) then 
		for k, buffno in pairs (groupbuffsshown) do
			local effect = 'buff' .. buffno;
			local j = nil;
			if ( BC_buffs_in_effect_by_name[effect] ) then
				j = BC_buffs_in_effect_by_name[effect];
			end
			if j ~= nil then
				BC_buffs_in_effect[j].fade = true;
			end
		end  
	elseif ( BC_class == BENECAST_STRINGS.CLASS_PALADIN and table.getn(groupblessingsshown) > 0 ) then
		for k, blessingno in pairs (groupblessingsshown) do 
			local effect = 'buff' .. blessingno;
			local j = nil;
			if ( BC_buffs_in_effect_by_name[effect] ) then
				j = BC_buffs_in_effect_by_name[effect];
			else
				effect = 'partybuff' .. blessingno;
				if ( BC_buffs_in_effect_by_name[effect] ) then
					j = BC_buffs_in_effect_by_name[effect];
				end
			end
			if j ~= nil then
				BC_buffs_in_effect[j].fade = true;
			end
		end  
	elseif ( BC_class == BENECAST_STRINGS.CLASS_MAGE and wardshown ) then
		if ( BC_buffs_in_effect_by_name['selfbuff12'] ) then
			local j = BC_buffs_in_effect_by_name['selfbuff12'];
			BC_buffs_in_effect[j].fade = true;
		end
		if ( BC_buffs_in_effect_by_name['selfbuff13'] ) then
			local j = BC_buffs_in_effect_by_name['selfbuff13'];
			BC_buffs_in_effect[j].fade = true;
		end
	end
	
	--Show Swiftmend if applicable
	buttonstochange['instant2'].add = true;
	if unitisfriend then
		if ( BC_class == BENECAST_STRINGS.CLASS_DRUID and (activeshapeshiftid == 0 or activeshapeshiftid == 6) ) then
			buttonstochange['instant2'].show = 
				( BC_buffs_in_effect[46][buff_attrib] or BC_buffs_in_effect[47][buff_attrib] );
			if not ( BC_buffs_in_effect[46][buff_attrib] or BC_buffs_in_effect[47][buff_attrib] ) then
				buttonstochange['instant2'].reason = 'No HoT active';
			end
		else
			buttonstochange['instant2'].show = false;
			if BC_class ~= BENECAST_STRINGS.CLASS_DRUID then
				buttonstochange['instant2'].reason = 'Not a Druid';
			else
				buttonstochange['instant2'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
			end
		end
	end
	
	-- Show buff buttons of buffs not in effect if possible
	for j in pairs (BC_buffs_in_effect) do
		-- Some buttons should not show on others
		-- Selfbuffs on others and buffparty on those outside of the party/raid
		-- Groupbuffs such as Prayer of Fortitude should only be shown on the target/player
		if not BC_buffs_in_effect[j].notabuff then
			-- For the target, always show the button
			if member == 'target' and not string.find(BC_buffs_in_effect[j].name,'self') then
				buttonstochange[BC_buffs_in_effect[j].name].add = true;
				buttonstochange[BC_buffs_in_effect[j].name].show = false;
			end
			if ( string.find(BC_buffs_in_effect[j].name, 'groupbuff') and
			     not ( member == 'target' or member == 'player' ) ) then
				-- Do nothing
			elseif ( string.find(BC_buffs_in_effect[j].name, 'self') and member ~= 'player' ) then
				-- Do nothing
			elseif ( string.find(BC_buffs_in_effect[j].name, 'party') and
			          not ( UnitIsUnit(member,'player') or UnitInParty(member) or 
				        BeneCast_UnitInRaid(member) or BeneCast_UnitIsPartyPet(member) or 
					BeneCast_UnitIsRaidPet(member) ) ) then
				-- Do nothing
			else
				-- First show the button
				buttonstochange[BC_buffs_in_effect[j].name].add = true;
				local allowedinthisform = true;
				if BC_class == BENECAST_STRINGS.CLASS_DRUID and activeshapeshiftid ~= 0 then
					allowedinthisform = false;
					if activeshapeshiftid == 5 then
						if ( BC_buffs_in_effect[j].name == 'selfbuff2' ) then
							allowedinthisform = true;
						end
						if ( BC_buffs_in_effect[j].name == 'selfbuff3' ) then
							allowedinthisform = true;
						end
						if ( BC_buffs_in_effect[j].name == 'selfbuff4' ) then
							allowedinthisform = true;
						end
						if ( BC_buffs_in_effect[j].name == 'buff2' ) then
							allowedinthisform = true;
						end
					end
					if activeshapeshiftid == 6 then
						if ( BC_buffs_in_effect[j].name == 'selfbuff1' ) then
							allowedinthisform = true;
						end
						if ( BC_buffs_in_effect[j].name == 'selfbuff4' ) then
							allowedinthisform = true;
						end
						if ( BC_buffs_in_effect[j].name == 'buff3' ) then
							allowedinthisform = true;
						end
					end
				end				
				if allowedinthisform then --WINTROW.2 & 4
					-- Then see about fading it
					local fade = BC_buffs_in_effect[j].fade;
					if not fade then
						-- Make this button fade if it is for PW:S and the effect Weakened Soul is on the target
						if ( BC_class == BENECAST_STRINGS.CLASS_PRIEST and BC_buffs_in_effect[j].name == 'partybuff1' and weakenedsoul ) then
							fade = true;
							buttonstochange[BC_buffs_in_effect[j].name].reason = 'Weakened Soul is active';
						end
					else
						buttonstochange[BC_buffs_in_effect[j].name].reason = 'Buff is active';
					end
					buttonstochange[BC_buffs_in_effect[j].name].show = not fade;
				else
					buttonstochange[BC_buffs_in_effect[j].name].show = false;
					buttonstochange[BC_buffs_in_effect[j].name].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
				end 
			end
		else
			--if BC_buffs_in_effect[j].name == 'instant' then
			--	PutDebugMsg('Skipping "instant" as a buff-button');
			--end
		end
	end
	
	-- Show Weapon Enchants for Shaman and Priests
	if ( member == 'player') then
		if ( BC_class == BENECAST_STRINGS.CLASS_SHAMAN ) then
			-- Show all weaponenchants
			buttonstochange['weaponenchant1'].add = true;
			buttonstochange['weaponenchant2'].add = true;
			buttonstochange['weaponenchant3'].add = true;
			buttonstochange['weaponenchant4'].add = true;
			-- See about fading them
			buttonstochange['weaponenchant1'].show = 
				( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_ROCKBITER );
			if ( BC_weaponenchant == BENECAST_STRINGS.WEAPONENCHANT_ROCKBITER ) then
				buttonstochange['weaponenchant1'].reason = 'Buff is active';
			end
			buttonstochange['weaponenchant1'].show = 
				( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_FLAMETONGUE );
			if ( BC_weaponenchant == BENECAST_STRINGS.WEAPONENCHANT_FLAMETONGUE ) then
				buttonstochange['weaponenchant2'].reason = 'Buff is active';
			end
			buttonstochange['weaponenchant1'].show = 
				( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_FROSTBRAND );
			if ( BC_weaponenchant == BENECAST_STRINGS.WEAPONENCHANT_FROSTBRAND ) then
				buttonstochange['weaponenchant3'].reason = 'Buff is active';
			end
			buttonstochange['weaponenchant1'].show = 
				( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_WINDFURY );
			if ( BC_weaponenchant == BENECAST_STRINGS.WEAPONENCHANT_WINDFURY ) then
				buttonstochange['weaponenchant4'].reason = 'Buff is active';
			end
		end
		if ( BC_class == BENECAST_STRINGS.CLASS_PRIEST ) then
			buttonstochange['weaponenchant1'].add = true;
			buttonstochange['weaponenchant1'].show = 			
				( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_FEEDBACK );
			if ( BC_weaponenchant == BENECAST_STRINGS.WEAPONENCHANT_FEEDBACK ) then
				buttonstochange['weaponenchant1'].reason = 'Buff is active';
			end
		end
	end
	
	-- Show Cure buttons as temporary buttons
	
	if ( BC_buffs_in_effect[48].fade ) then
		poisonshow = true; --for flashing the poison button while Abolish Poison is active
	end
	
	if curemagic then
		magicshow = true;
	end
		
	-- Show buttons for effects that are removable once
	-- First show the buttons
	buttonstochange['poison'].add = true;
	buttonstochange['disease'].add = true;
	buttonstochange['disease2'].add = true;
	buttonstochange['magic'].add = true;
	buttonstochange['curse'].add = true;
	if unitisfriend then
		buttonstochange['poison'].show = (poisonshow and activeshapeshiftid == 0);
		if activeshapeshiftid ~= 0 then
			buttonstochange['poison'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
		elseif not poisonshow then
			buttonstochange['poison'].reason = 'No Dispellable debuff found';
		end
		buttonstochange['disease'].show = diseaseshow;
		if not diseaseshow then
			buttonstochange['disease'].reason = 'No Dispellable debuff found';
		end
		buttonstochange['disease2'].show = diseaseshow;
		if not diseaseshow then
			buttonstochange['disease2'].reason = 'No Dispellable debuff found';
		end
		buttonstochange['magic'].show = magicshow;
		if not diseaseshow then
			buttonstochange['magic'].reason = 'No Dispellable debuff found';
		end
		buttonstochange['curse'].show = (curseshow and activeshapeshiftid == 0);
		if activeshapeshiftid ~= 0 then
			buttonstochange['curse'].reason = 'activeshapeshiftid = ' .. activeshapeshiftid;
		elseif not poisonshow then
			buttonstochange['curse'].reason = 'No Dispellable debuff found';
		end
	end
	
	if disableall then
		for spelltype in pairs(buttonstochange) do
			if spelltype == 'res' and showres then
				-- Do Nothing
			elseif spelltype == 'magic' and curemagic then
				-- Do Nothing
			else
				buttonstochange[spelltype].show = false;
				buttonstochange[spelltype].reason = 'Not applicable';
			end
		end
	end
	
	local showallbuttons;
	if InCombatLockdown() or BC_ConsiderInCombat then
		showallbuttons = true
	else
		showallbuttons = BeneCastConfig.ShowAllButtons;
	end
	
	if not showallbuttons then
		for spelltype in pairs(buttonstochange) do
			if not buttonstochange[spelltype].show then
				buttonstochange[spelltype].add = false;
			end
		end
	end
	
	local firstfreebutton = 1;
	if panel:CanChangeProtectedState() then
		-- Hide all buttons before redrawing them
		for x = firstfreebutton, BENECAST_MAX_NUMBER_OF_BUTTONS do
			BeneCast_HideButton(x, id, member);
		end
	end
	for i, spelltype in pairs(BC_spellsort) do
		if panel:CanChangeProtectedState() then
			if buttonstochange[spelltype].add then
				if ( BeneCast_ShowButton(spelltype, firstfreebutton, id, member) ) then
					firstfreebutton = firstfreebutton+1;
				end
			end
		end
		if BC_buttons_by_spell[member] then
			if disableall or not buttonstochange[spelltype].show then
				BeneCast_SetButtonFade(BC_buttons_by_spell[member][spelltype],false,buttonstochange[spelltype].reason);
			else
				BeneCast_SetButtonFade(BC_buttons_by_spell[member][spelltype],true,'');
			end
		end
	end
	
	--WINTROW.6 START
	--Check for flashing
	local iterator = 1; -- First clean out the flash-buttons of this panel, ready to be re-added if still active
	while iterator <= table.getn(BC_buttons_that_flash) do
		if string.find(BC_buttons_that_flash[iterator],'BeneCastPanel' .. id .. 'Button') then
			table.remove(BC_buttons_that_flash,iterator);
		else
			iterator = iterator + 1;
		end
	end
	if BC_buttons_by_spell[member] then
		if ( BC_class == BENECAST_STRINGS.CLASS_PRIEST ) then
			if BC_buffs_in_effect[46].fade then
				PutDebugMsg('"instant" is in effect');
				if BC_buttons_by_spell[member]['instant'] then
					if BeneCastConfig.FlashAsFade then
						getglobal(BC_buttons_by_spell[member]['instant'] .. 'Fade'):Show();
					else
						table.insert(BC_buttons_that_flash,BC_buttons_by_spell[member]['instant']);
					end
				else
					PutDebugMsg('BC_buttons_by_spell[' .. member .. '][instant] is nil');
				end
			end
		end
		if ( BC_class == BENECAST_STRINGS.CLASS_DRUID ) then
			if BC_buffs_in_effect[46].fade then
				PutDebugMsg('"instant" is in effect');				
				if BC_buttons_by_spell[member]['instant'] then
					if BeneCastConfig.FlashAsFade then
						getglobal(BC_buttons_by_spell[member]['instant'] .. 'Fade'):Show();
					else
						table.insert(BC_buttons_that_flash,BC_buttons_by_spell[member]['instant']);
					end
				else
					PutDebugMsg('BC_buttons_by_spell[' .. member .. '][instant] is nil');
				end
			end
			if BC_buffs_in_effect[47].fade then
				PutDebugMsg('"emergency" is in effect');
				if BC_buttons_by_spell[member]['emergency'] then
					if BeneCastConfig.FlashAsFade then
						getglobal(BC_buttons_by_spell[member]['emergency'] .. 'Fade'):Show();
					else
						table.insert(BC_buttons_that_flash,BC_buttons_by_spell[member]['emergency']);
					end
				else
					PutDebugMsg('BC_buttons_by_spell[' .. member .. '][emergency] is nil');
				end
			end
			if BC_buffs_in_effect[48].fade then
				PutDebugMsg('"poison" is in effect');
				if BC_buttons_by_spell[member]['poison'] then
					if BeneCastConfig.FlashAsFade then
						getglobal(BC_buttons_by_spell[member]['poison'] .. 'Fade'):Show();
					else
						table.insert(BC_buttons_that_flash,BC_buttons_by_spell[member]['poison']);
					end
				else
					PutDebugMsg('BC_buttons_by_spell[' .. member .. '][poison] is nil');
				end
			end
		end
		if ( BC_class == BENECAST_STRINGS.CLASS_SHAMAN ) then
			if BC_buffs_in_effect[48].fade then
				PutDebugMsg('"poison" is in effect');
				if BC_buttons_by_spell[member]['poison'] then
					if BeneCastConfig.FlashAsFade then
						getglobal(BC_buttons_by_spell[member]['poison'] .. 'Fade'):Show();
					else
						table.insert(BC_buttons_that_flash,BC_buttons_by_spell[member]['poison']);
					end
				else
					PutDebugMsg('BC_buttons_by_spell[' .. member .. '][poison] is nil');
				end
			end
		end
	end
	--WINTROW.6 STOP

	-- Hide all buttons that shouldn't be shown
	if panel:CanChangeProtectedState() then
		for x = firstfreebutton, BeneCastConfig.ButtonNumber do
			BeneCast_HideButton(x, id, member);
		end
	end
	
	-- There is not reason that we would hide this later, so we can check this panel each
	-- 	time we have a reason to suspect a change.
	BC_attached_targets[id] = true;
	
	-- If there are no buttons to be shown, hide the frame
	if ( firstfreebutton == 1 and panel:CanChangeProtectedState() ) then
		panel:Hide();
		return;
	end
	
	-- Resize the BeneCastPanelFrame based on the number of buttons
	BeneCast_ResizeButtonsFor(id);
		
end


-- *****************************************************************************
-- BeneCastPanelMenu functions
-- *****************************************************************************

function BeneCastPanelDropDown_Initialize()
	
	local info;
	-- Add Title Button
	info = {};
	if ( UnitExists(BC_targets[this:GetID()]) ) then
		info.text = UnitName(BC_targets[this:GetID()]);
	else
		info.text = BENECAST_STRINGS.TEXT_UNKNOWN
	end
	info.isTitle = 1;
	UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);
	
	-- Add Lock Button
	info = {};
	info.text = BENECAST_STRINGS.TEXT_LOCKED;
	info.value = this:GetID();
	info.checked = this.locked;
	info.func = BeneCastPanel_SetLock;
	UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);
	
	-- Add Orientation Button
	info = {};
	info.text = 'Change Orientation';
	info.value = this:GetID() .. 'orient';
	info.func = BeneCastPanel_SetOrientation;
	UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);

end

function BeneCastPanel_SetLock() 

	local frame = getglobal('BeneCastPanel' .. this.value);
	frame.locked = not frame.locked;
	
end

function BeneCastPanel_SetOrientation()

	local id = nil;
	_, _, id = string.find(this.value,'(.+)orient');
	PutDebugMsg('BeneCast_SetOrientation() called');
	if not BeneCastConfig['BeneCastPanel' .. id] then
		BeneCastConfig['BeneCastPanel' .. id] = {};
	end
	if BeneCastConfig['BeneCastPanel' .. id]['Orientation'] == 'down' then
		BeneCastConfig['BeneCastPanel' .. id]['Orientation'] = 'right';
	else
		BeneCastConfig['BeneCastPanel' .. id]['Orientation'] = 'down';
	end
	PutDebugMsg("New value BeneCastConfig['BeneCastPanel' .. id]['Orientation'] = " ..
	            BeneCastConfig['BeneCastPanel' .. id]['Orientation']);
	BeneCast_OrientButtons(id);

end

function BeneCastPanelMenu_OnLoad()

	UIDropDownMenu_Initialize(this, BeneCastPanelDropDown_Initialize, 'MENU');

end

-- *****************************************************************************
-- BeneCastPanel functions
-- *****************************************************************************

function BeneCastPanel_OnLoad()

	-- set our default colors
	this:SetBackdropBorderColor(TOOLTIP_DEFAULT_COLOR.r, TOOLTIP_DEFAULT_COLOR.g, TOOLTIP_DEFAULT_COLOR.b);
	this:SetBackdropColor(TOOLTIP_DEFAULT_BACKGROUND_COLOR.r, TOOLTIP_DEFAULT_BACKGROUND_COLOR.g, TOOLTIP_DEFAULT_BACKGROUND_COLOR.b);
	this.locked = 1;

end

function BeneCastPanel_OnMouseDown()

	if ( ( ( not this.locked ) or ( this.locked == 0 ) or BeneCastConfig.Unlock ) and ( arg1 == 'LeftButton' ) ) then
		this:StartMoving();
		this.isMoving = true;
	end
	
end

--WINTROW.6 START
--Moved this from the XML-code so I can add code to save the panel's locations
function BeneCastPanel_OnMouseUp()

	if ( this.isMoving ) then
		this:StopMovingOrSizing();
		this.isMoving = false;
		local framename = 'BeneCastPanel' .. this:GetID();
		if not BeneCastConfig[framename] then
			BeneCastConfig[framename] = {
				['Y'] = 0,
				['X'] = 0,
				['Point'] = 'TOPLEFT',
				['RelativePoint'] = 'TOPLEFT',
				['Frame'] = '',
				['Moved'] = true,
			};
		else
			BeneCastConfig[framename]['Point'] = 'TOPLEFT';
			BeneCastConfig[framename]['RelativePoint'] = 'TOPLEFT';
		end
		local parent = this:GetParent();
		BeneCastConfig[framename]['Frame'] = parent:GetName();
		-- The values stored are unscaled values (real pixels)
		-- When being applied they will need to be adjusted to the scale between the frame and his parent
		BeneCastConfig[framename]['X'] = ( this:GetLeft() * this:GetEffectiveScale() ) - 
						 ( parent:GetLeft() * parent:GetEffectiveScale() ) + 0.5;
		BeneCastConfig[framename]['X'] = math.floor(BeneCastConfig[framename]['X']);
		BeneCastConfig[framename]['Y'] = ( this:GetTop() * this:GetEffectiveScale() ) -
						 ( parent:GetTop() * parent:GetEffectiveScale() ) + 0.5;
		BeneCastConfig[framename]['Y'] = math.floor(BeneCastConfig[framename]['Y']);
		BeneCastConfig.SnapTo = 'CUSTOM';
		BeneCastConfig[framename]['Moved'] = true;
		UIDropDownMenu_SetSelectedValue(BeneCastSnapToMenu, BeneCastConfig.SnapTo);
		-- Make sure BeneCast does all the moving around here... Thanks Snidely!
		if this:IsUserPlaced() then
			this:SetUserPlaced(false);
		end
		BeneCast_AttachPanelsFromVar(this:GetID());
		BeneCastUpdateCustomEditBoxes();
	end
	
end

function BeneCastPanel_OnEnter()
	if BeneCastConfig.Debug then
		local id, unitname, target, framename;
		id = this:GetID();
		framename = this:GetName();
		if not id then
			local idstr;
			_, _, idstr = string.find(framename,'BeneCastPanel(.+)');
			id = tonumber(idstr);
		end
		if id then
			target = BeneCast_GetUnitIDFromPanelID(id);
			if target then
				unitname = UnitName(target);
			end
		end
		if not target then
			target = 'nil';
		end
		if not unitname then
			unitname = 'nil';
		end
		if not id then
			id = '?';
		end
		PutDebugMsg('Mousing over ' .. framename .. ', targetid = ' .. target .. ', name is ' .. unitname);
	end
end
--WINTROW.6 STOP

-- *****************************************************************************
-- BeneCastOptionFrameTab functions
-- *****************************************************************************

function BeneCastOptionFrameTab_OnClick()

	-- Show the correct subframe based on which tab was clicked
	-- Hide all subframes that do not have matching ID numbers, show the matching subframe
	for subframe, subframename in pairs (BeneCastOptionFrameSubframes) do
		if this:GetID() == getglobal(subframename):GetID() then
			getglobal(subframename):Show();
		else
			getglobal(subframename):Hide();
		end
	end
	
	-- Play a sound for player feedback
	PlaySound('igCharacterInfoTab');
	
	-- Set the glowing checked status for the correct tab
	for x in pairs (BeneCastOptionFrameSubframes) do
		if ( x == this:GetID() ) then
			this:Disable()
			this:SetChecked(1);
		else
			getglobal('BeneCastOptionFrameTab' .. x):Enable();
			getglobal('BeneCastOptionFrameTab' .. x):SetChecked(nil);
		end
	end
	
end

-- *****************************************************************************
-- BeneCastOptionsCheckButton functions
-- *****************************************************************************

function BeneCastOptionsCheckButton_OnClick()

	local dontredraw = nil;

	-- Set the BeneCastConfig for the option to take on the value of the checkbutton
	for index, value in pairs (BeneCastOptionFrameCheckButtons) do
		if ( value.index == this:GetID() ) then
			BeneCastConfig[value.cvar] = this:GetChecked();
			-- If this is the tooltip option check to see if it's enabled
			-- If not disable the tooltip name option checkbutton
			if ( value.index == 5 ) then
				if ( this:GetChecked() ) then
					getglobal('BeneCastOptionCheckButton6'):Enable();
				else
					getglobal('BeneCastOptionCheckButton6'):Disable();
				end
			-- If this is the notification option check to see if it's enabled
			-- If not disable all sub notification options
			elseif ( value.index == 8 ) then
				if ( this:GetChecked() ) then
					getglobal('BeneCastOptionCheckButton9'):Enable();
					getglobal('BeneCastOptionCheckButton10'):Enable();
					getglobal('BeneCastOptionCheckButton11'):Enable();
					getglobal('BeneCastOptionCheckButton13'):Enable();
					getglobal('BeneCastOptionCheckButton14'):Enable();
					getglobal('BeneCastOptionCheckButton15'):Enable();
					getglobal('BeneCastOptionCheckButton16'):Enable();
					getglobal('BeneCastOptionCheckButton17'):Enable();
					getglobal('BeneCastOptionCheckButton18'):Enable();
					getglobal('BeneCastOptionCheckButton19'):Enable();
					getglobal('BeneCastOptionCheckButton21'):Enable();
				else
					getglobal('BeneCastOptionCheckButton9'):Disable();
					getglobal('BeneCastOptionCheckButton10'):Disable();
					getglobal('BeneCastOptionCheckButton11'):Disable();
					getglobal('BeneCastOptionCheckButton13'):Disable();
					getglobal('BeneCastOptionCheckButton14'):Disable();
					getglobal('BeneCastOptionCheckButton15'):Disable();
					getglobal('BeneCastOptionCheckButton16'):Disable();
					getglobal('BeneCastOptionCheckButton17'):Disable();
					getglobal('BeneCastOptionCheckButton18'):Disable();
					getglobal('BeneCastOptionCheckButton19'):Disable();
					getglobal('BeneCastOptionCheckButton21'):Disable();
				end
			-- If this is the heal notification option check to see if it's enabled
			-- If not disable all heal max notification
			elseif ( value.index == 16 ) then
				if ( this:GetChecked() ) then
					getglobal('BeneCastOptionCheckButton17'):Enable();
				else
					getglobal('BeneCastOptionCheckButton17'):Disable();
				end
			-- If this is the hide minimap button option check to see if it's enabled
			-- If so then hide the minimap button
			elseif ( value.index == 4 ) then
				if ( this:GetChecked() ) then
					getglobal('BeneCastMinimapButton'):Hide();
				else
					getglobal('BeneCastMinimapButton'):Show();
				end	
			--If Damage Based Healing is turned off --> Priest Heals are seperated
			--If Damage Based Healing is turned on  --> Priest Heals are consolidated
			elseif ( value.index == 1 ) then
				-- Reset most of the tables for reloading of spells and reload the spell data
				-- This will split the Priest Heal buttons
				BeneCast_ResetData();
				BeneCast_UpdateHealingBonuses(); --Recalc healingbonuses
			--Allow Raidpets to be clicked only if Raidmembers is checked
			elseif ( value.index == 24 ) then
				local cbutton = getglobal('BeneCastOptionCheckButton25');
				if ( this:GetChecked() ) then
					cbutton:Enable();
				else
					BeneCastConfig.ShowRaidPets = false;
					cbutton:SetChecked(BeneCastConfig.ShowRaidPets);
					cbutton:Disable();
				end
				BC_UpdateRaid = true;
				dontredraw = true;
			--Redraw raidbuttons to show/hide raidpets
			elseif ( value.index == 25 ) then
				BC_UpdateRaid = true;
				dontredraw = true;
			--Reverse HoTs -> Recalc healingbonuses
			elseif ( value.index == 26 ) then
				BeneCast_UpdateHealingBonuses();
			end
		end
	end
	-- Play a sound for player feedback
	if ( this:GetChecked() ) then
		PlaySound('igMainMenuOptionCheckBoxOff');
	else
		PlaySound('igMainMenuOptionCheckBoxOn');
	end
	
	-- Redraw everyone's buttons
	if not dontredraw then
		for i in pairs (BC_targets) do
			BeneCast_UpdateButtons(i);
		end
	end

end

-- *****************************************************************************
-- BeneCastProfileCheckButton functions
-- *****************************************************************************

function BeneCastProfileCheckButton_OnClick()

	local profilebox = getglobal('BeneCastProfileName' .. this.var2);
	if profilebox:GetText() then
		if BeneCastProfiles[profilebox:GetText()] then
			if this.var1 == 'profilecombat' then
				for i = 1, 5 do
					if i ~= this.var2 then
						getglobal('BeneCastCombatCheck' .. i):SetChecked(false);
					end
				end
				BeneCastSetup['PrefferedInCombatProfile'] = profilebox:GetText();
			elseif this.var1 == 'profileoutofcombat' then
				for i = 1, 5 do
					if i ~= this.var2 then
						getglobal('BeneCastOutOfCombatCheck' .. i):SetChecked(false);
					end
				end
				BeneCastSetup['PrefferedOutOfCombatProfile'] = profilebox:GetText();
			end
			UpdateCurrentProfile();
		end
	end
				
end

function BeneCast_UpdateProfileCheckButtons()

	local profindex = 1;
	for profilename in pairs(BeneCastProfiles) do
		local profilebox = getglobal('BeneCastProfileName' .. profindex);
		profilebox:SetText(profilename);
		profindex = profindex + 1;
	end
	for i = profindex, 5 do
		local profilebox = getglobal('BeneCastProfileName' .. i);
		profilebox:SetText('');
	end

	for i = 1, 5 do
		local profilebox = getglobal('BeneCastProfileName' .. i);
		local checkbutton;
		checkbutton = getglobal('BeneCastCombatCheck' .. i);
		checkbutton:SetChecked(profilebox:GetText() == BeneCastSetup['PrefferedInCombatProfile']);
		checkbutton = getglobal('BeneCastOutOfCombatCheck' .. i);
		checkbutton:SetChecked(profilebox:GetText() == BeneCastSetup['PrefferedOutOfCombatProfile']);
	end
end

function UpdateCurrentProfile()

	-- Save previous config
	BeneCastProfiles[BC_CurrentProfile] = BeneCastConfig;
	if BC_ConsiderInCombat then
		BC_CurrentProfile = BeneCastSetup['PrefferedInCombatProfile'];
	else
		BC_CurrentProfile = BeneCastSetup['PrefferedOutOfCombatProfile'];
	end
	if BeneCastProfiles[BC_CurrentProfile] then
		BeneCastConfig = BeneCastProfiles[BC_CurrentProfile];
		BeneCast_LoadOptions();
		for i,attached in pairs (BC_attached_targets) do
			if attached then
				BeneCast_UpdateButtons(i);
			else
				PutDebugMsg('Not updating "'..BC_targets[i]..'" because it was not previously attached.');
			end
		end
	end		

end

-- *****************************************************************************
-- BeneCastSpellCheckButton functions
-- *****************************************************************************

function BeneCastSpellCheckButton_OnClick()

	-- Put some things in a local in the beginning for efficiency (thnx Shimavak)
	local spellname = getglobal( this:GetName() .. 'Text' ):GetText();
	local parentid = this:GetParent():GetID();
	-- Set the BeneCastConfig for the option to take on the value of the checkbutton
	local spelltype;
	if ( spellname == BENECAST_STRINGS.TEXT_ASSIST ) then
		spelltype = 'assist';
	else
		spelltype = BeneCast_SpellTypes[BC_class][spellname];
	end
	if ( BeneCastConfig['PlayerAsDefault'] and parentid == 1 ) then
		for i = 1, 10 do
			BeneCastConfig[i][spelltype] = this:GetChecked();
			getglobal(BeneCastOptionFrameSubframes[i] .. 'Button' .. this:GetID()):SetChecked(this:GetChecked());
		end
	else
		BeneCastConfig[parentid][spelltype] = this:GetChecked();
	end
	-- Play a sound for player feedback
	if ( this:GetChecked() ) then
		PlaySound('igMainMenuOptionCheckBoxOff');
	else
		PlaySound('igMainMenuOptionCheckBoxOn');
	end
	
	-- Redraw everyone's buttons
	for i in pairs (BC_targets) do
		BeneCast_UpdateButtons(i);
	end

end

-- *****************************************************************************
-- BeneCastRaidCheckButton functions
-- *****************************************************************************

function BeneCastRaidCheckButton_OnClick()

	-- If this raid checkbutton has a name
	if ( getglobal(this:GetName() .. 'Name'):GetText() ~= '' ) then
		-- Set the RaidID of this member to on or off
		local name = getglobal(this:GetName() .. 'Name'):GetText();
		local unitid = BENECAST_RAID_ROSTER2[name];
		local idnum;
		_, _, idnum = string.find(unitid,'raid(.+)');
		BENECAST_RAID_LIST[name] = this:GetChecked();
		
		BeneCast_AttachRaidPanels(BeneCastConfig.RaidSnapTo);
		
		-- Update the buttons associated with the raid member and their pets
		BeneCast_UpdateButtonsForUnit('raid' .. idnum);
		BeneCast_UpdateButtonsForUnit('raidpet' .. idnum);
	end

end

function BeneCastRaidCheckButton_OnEnter()

	GameTooltip:SetOwner(this, 'ANCHOR_BOTTOMRIGHT');
	GameTooltip:SetText(BENECAST_STRINGS.TEXT_RAID_CHECKBUTTON_TOOLTIP, nil, nil, nil, nil, 1)

end

-- *****************************************************************************
-- BeneCastRaidGroup functions
-- *****************************************************************************

function BeneCastRaidGroup_OnClick()

	local name, rank, subgroup, level, class, fileName, zone, online, isDead;
	
	-- For all raid members/ids
	for i = 1, GetNumRaidMembers() do
		name, rank, subgroup, level, class, fileName, zone, online, isDead = GetRaidRosterInfo(i);
		-- Check this raid member to if their subgroup is the same as this subgroup
		if ( subgroup == this:GetID() ) then
			if ( name ) then
				-- Set the RaidID of this group member to on
				BENECAST_RAID_LIST[name] = true;
			end
		end
	end
	
	BC_UpdateRaid = true;

end

function BeneCastRaidGroup_OnLoad()

	getglobal(this:GetName() .. 'Name'):SetText(BENECAST_STRINGS.TEXT_RAID_GROUP .. this:GetID());

end

function BeneCastRaidGroup_OnEnter()

	GameTooltip:SetOwner(this, 'ANCHOR_BOTTOMRIGHT');
	GameTooltip:SetText(BENECAST_STRINGS.TEXT_RAID_GROUP_TOOLTIP, nil, nil, nil, nil, 1)

end

-- *****************************************************************************
-- BeneCastOptionsSlider functions
-- *****************************************************************************

function BeneCastOptionsSlider_OnValueChanged()

	-- Set the BeneCastConfig for the option to take on the value of the checkbutton
	for index, value in pairs (BeneCastOptionFrameSliders) do
		if ( value.index == this:GetID() ) then
			BeneCastConfig[value.cvar] = this:GetValue();
			getglobal('BeneCastOptionSlider'..value.index..'Value'):SetText(this:GetValue());
		end
		-- If the slider being changed is the buttonsize slider then resize the buttons
		if ( value.index == 1 ) then
			BeneCast_ResizeButtons();
		end
		-- If the slider being changed is the buttonnumber slider then redraw all buttons
		if ( value.index == 3 ) then
			for i in pairs (BC_targets) do
				BeneCast_UpdateButtons(i);
			end
		end
	end

end

-- *****************************************************************************
-- BeneCast_Tooltip functions
-- *****************************************************************************

function BeneCast_Tooltip_OnLoad()

	-- For some unknown reason, when doing reloadui, it will not parse the OnLoad from
	-- 	the XML file for BeneCast_Tooltip...so we're doing that here.
	this:SetOwner(WorldFrame, "ANCHOR_NONE");
end

-- *****************************************************************************
-- BeneCastPanelManager functions
-- *****************************************************************************

function BeneCastPanelManager_OnLoad()

	-- Set BC_class to the player's class
	BC_class = UnitClass('player');
	
	-- Register for Events
	this:RegisterEvent('ADDON_LOADED');
	this:RegisterEvent('PLAYER_ENTERING_WORLD');
	this:RegisterEvent('PLAYER_LEAVING_WORLD');
	this:RegisterEvent('VARIABLES_LOADED');
	this:RegisterEvent('SPELLS_CHANGED');
	this:RegisterEvent('SPELLCAST_FAILED');
	this:RegisterEvent('SPELLCAST_INTERRUPTED');
	this:RegisterEvent('SPELLCAST_STOP');
	this:RegisterEvent('SPELLCAST_START');
	this:RegisterEvent('UNIT_SPELLCAST_CHANNEL_START');
	this:RegisterEvent('UNIT_SPELLCAST_CHANNEL_STOP');
	this:RegisterEvent('PARTY_MEMBERS_CHANGED');
	this:RegisterEvent('PLAYER_TARGET_CHANGED');
	this:RegisterEvent('PLAYER_FOCUS_CHANGED');
	this:RegisterEvent('UNIT_AURA');
	this:RegisterEvent('PLAYER_AURAS_CHANGED');
	this:RegisterEvent('UNIT_AURASTATE');
	this:RegisterEvent('UPDATE_SHAPESHIFT_FORMS');
	this:RegisterEvent('PLAYER_ALIVE');
	this:RegisterEvent('PLAYER_DEAD');
	this:RegisterEvent('PLAYER_UNGHOST');
	this:RegisterEvent('RAID_ROSTER_UPDATE');
	this:RegisterEvent('UNIT_HEALTH');
	this:RegisterEvent('UNIT_PET');
	this:RegisterEvent('UNIT_TARGET');
	this:RegisterEvent('PLAYER_REGEN_DISABLED');
	this:RegisterEvent('PLAYER_REGEN_ENABLED');

end

function BeneCastPanelManager_OnEvent()

	if ( event == 'ADDON_LOADED' and arg1 == 'BeneCast' ) then
		-- Reset the data if there BeneCast is severly outdated
		if ( not BeneCastConfig or not BeneCastSetup or not BeneCastProfiles or
		     table.getn(BeneCastProfiles) == 0 or
		     not BeneCastConfig.Version or BeneCastConfig.Version < BENECAST_VERSION ) then
			BeneCast_ResetProfiles();
		end
		-- Populating all of our lists of constants now so we don't have to do it repeatedly.
		for i in pairs (BC_targets) do
			BC_attached_targets[i] = false;
		end
		
		for i,buffInfo in pairs (BC_buffs_in_effect) do
			BC_buffs_in_effect_by_name[buffInfo.name] = i;
		end
		
		-- Load up spell data once AddOn has been loaded
		BeneCast_ResetData();
		
		-- In older versions there was no right/bottom
		if ( BeneCastConfig.RaidSnapTo == 'CT_RAID' ) then
			BeneCastConfig.RaidSnapTo = 'CT_RAIDRIGHT';
		end
		
		--Code from BeneCastOptionFrame
		BeneCast_LoadOptions();
		BeneCast_UpdateProfileCheckButtons();
		BeneCast_ResizeButtons();
		BeneCast_ClearRaid();
		
		BENECAST_RAID_LIST = {};
		
		--If BonusScanner was loaded before BeneCast
		if IsAddOnLoaded('BonusScanner') then
			if BC_BonusScanner_HookFunction == nil then
				BeneCast_UpdateHealingBonuses();
				BC_BonusScanner_HookFunction = BonusScanner_Update;
				BonusScanner_Update = BeneCast_UpdateHealingBonuses;
			end
		end
		
	elseif ( event == 'ADDON_LOADED' and ( arg1 == 'Blizzard Raid UI' or arg1 == 'CT_RaidAssist' ) ) then
		if arg1 == 'CT_RaidAssist' then
			--Hook into CT_RA's update function
			if BC_Old_CTRA_UpdateFunction == nil then
				BC_Old_CTRA_UpdateFunction = CT_RA_UpdateRaidGroup;
				CT_RA_UpdateRaidGroup = BC_UpdateRaidGroup;
			end
		end
			
		BC_UpdateRaid = true;
	
	--Hook into BonusScanner's Update-function
	elseif ( event == 'ADDON_LOADED' and arg1 == 'BonusScanner' ) then
		if BC_BonusScanner_HookFunction == nil then
			BeneCast_UpdateHealingBonuses();
			BC_BonusScanner_HookFunction = BonusScanner_Update;
			BonusScanner_Update = BeneCast_UpdateHealingBonuses;
		end
	
	elseif ( event == 'ADDON_LOADED' ) then
		-- Load up the variables for the snapping if empty
		if BeneCastConfig.SnapTo ~= 'CUSTOM' then
			BeneCast_SnapPanels(arg1);
		end
		
	elseif ( event == 'VARIABLES_LOADED' ) then
		BeneCast_SnapPanels(BeneCastConfig.SnapTo);
		-- Wait with the snapping till everything is loaded
		BC_AttachPartyFrames = true;
	
	-- The following is really, really confusing
	-- If SPELLCAST_FAILED then a BeneCast spell may have failed
	elseif ( event == 'SPELLCAST_FAILED' ) then
		-- A BeneCast spell failed if a BC_spellstarted and BC_sentmessage
		if ( BC_spellstarted and BC_sentmessage ) then
			BeneCast_PartyNotifyFail('SPELLCAST_FAILED');
		end
		-- Clear out everything used for party notification just in case
		BC_spellstarted = nil;
		BC_spellstopped = nil;
		BC_spelltarget = nil;
		BC_spellbeingcast = nil;
		BC_spellismax = nil;
		BC_sentmessage = nil;
		
	-- If SPELLCAST_INTERRUPTED then a BeneCast spell may have been interrupted
	elseif ( event == 'SPELLCAST_INTERRUPTED' ) then
		-- A BeneCast spell was interrupted by the player if BC_spellstopped and BC_sentmessage
		if ( BC_spellstopped and BC_sentmessage ) then
			BeneCast_PartyNotifyFail('SPELLCAST_CANCELED');
		end
		-- A BeneCast spell was interrupted by something else if BC_spellstarted and BC_sentmessage
		if ( BC_spellstarted and BC_sentmessage ) then
			BeneCast_PartyNotifyFail('SPELLCAST_INTERRUPTED');
		end
		-- Clear out everything just in case
		BC_spellstarted = nil;
		BC_spellstopped = nil;
		BC_spelltarget = nil;
		BC_spellbeingcast = nil;
		BC_spellismax = nil;
		BC_sentmessage = nil;
		
	elseif ( event == 'SPELLCAST_STOP' or event == 'UNIT_SPELLCAST_CHANNEL_STOP' ) then
		-- If BC_spellbeingcast and notification is on then this is an instant spell and you should do notification
		if ( BC_spellbeingcast and BeneCastConfig.Notification ) then
			-- BC_sentmessage will be set to true of BeneCast_PartyNotify sends a message
			BC_sentmessage = BeneCast_PartyNotify(BC_spelltarget, BC_spellbeingcast, BC_spellismax, 0);
		-- Otherwise set _setmessage to false
		else
			BC_sentmessage = nil
		end
		-- If BC_spellstarted, then a BeneCast spell was started and you should set BC_sentmessage and BC_spellstopped to true to indicate that the spell was interrupted by the user
		if ( BC_spellstarted ) then
			BC_sentmessage = true;
			BC_spellstopped = true;
			BC_spellstarted = nil;
		else
			-- Clear this, just in case
			BC_spellstopped = nil;
		end
		-- Clear these out just in case
		BC_spelltarget = nil;
		BC_spellbeingcast = nil;
		BC_spellismax = nil;
		
	elseif ( event == 'SPELLCAST_START' or event == 'UNIT_SPELLCAST_CHANNEL_START' ) then
		-- If the spell being cast exists in BeneCast_SpellTypes, notification is on, and there is a BC_spellbeingcast then do notification
		if ( BeneCast_SpellTypes[BC_class][arg1] and BeneCastConfig.Notification and BC_spellbeingcast ) then
			-- Set BC_sentmessage to true of BeneCast_PartyNotify sent a message
			BC_sentmessage = BeneCast_PartyNotify(BC_spelltarget, BC_spellbeingcast, BC_spellismax, arg2);
			-- Clear out BC_spelltarget, BC_spellbeingcast, and BC_spellismax so it isn't used by SPELLCAST_STOP
			BC_spelltarget = nil;
			BC_spellbeingcast = nil;
			BC_spellismax = nil;
			-- If you sent a message then BC_spellstarted to indicate failures/interruptions should send notifications
			if ( BC_sentmessage ) then
				BC_spellstarted = true;
			end
		else
			-- Clear out everything just in case
			BC_spellstarted = nil;
			BC_spellstopped = nil;
			BC_spelltarget = nil;
			BC_spellbeingcast = nil;
			BC_spellismax = nil;
			BC_sentmessage = nil;
		end
	
	-- We only want spell changes after entering the world
	elseif ( event == 'SPELLS_CHANGED' and arg1 == nil) then -- don't fire when user only browses spellbook
		
		-- Reset most of the tables for reloading of spells and reload the spell data
		BeneCast_ResetData();
		for i,attached in pairs (BC_attached_targets) do
			if attached then
				BeneCast_UpdateButtons(i);
			else
				PutDebugMsg('Not updating "'..BC_targets[i]..'" because it was not previously attached.');
			end
		end
	
	-- Don't bother with UNIT_AURA if the player can attack the unit
	elseif ( ( event == 'UNIT_AURA' or event == 'UNIT_AURASTATE' ) and not UnitCanAttack('player', arg1) ) then
		if BC_AttachPartyFrames then
			BeneCast_AttachPartyPanelsFromVar();
			BC_AttachPartyFrames = nil;
		end
		-- Redraw the buttons for the UnitID in arg1
		if BeneCastConfig.Debug then
			local msg = arg1;
			if not msg then
				msg = 'nil';
			end
			PutDebugMsg(event .. ' for ' .. msg);
		end
		BeneCast_UpdateButtonsForUnit(arg1);
		
	elseif ( event == 'PLAYER_AURAS_CHANGED' ) then
		if BC_AttachPartyFrames then
			BeneCast_AttachPartyPanelsFromVar();
			BC_AttachPartyFrames = nil;
		end
		BeneCast_UpdateButtonsForUnit('player');
	
	elseif ( event == 'PARTY_MEMBERS_CHANGED' ) then
		-- Redraw the buttons for all partymembers
		for i = 2,5 do
			BeneCast_UpdateButtons(i);
		end
		--Party pets too
		for i = 47,50 do
			BeneCast_UpdateButtons(i);
		end
	
	-- Fired when the pet of a unit changes
	elseif ( event == 'UNIT_PET' ) then
		if arg1 == 'player' then
			BeneCast_UpdateButtonsForUnit('pet');
		elseif string.find(arg1,'party') then
			local num;
			_, _, num = string.find(arg1,'party(.+)');
			if num then
				BeneCast_UpdateButtonsForUnit('partypet' .. num);
			end
		elseif string.find(arg1,'raid') then
			if BeneCastConfig.ShowRaidPets then
				BC_UpdateRaid = true;
			else
				local num;
				_, _, num = string.find(arg1,'raid(.+)');
				if num then
					BeneCast_UpdateButtonsForUnit('raidpet' .. num);
				end
			end
		end
		
	elseif ( event == 'PLAYER_TARGET_CHANGED' ) then
		if BC_AttachPartyFrames then
			BeneCast_AttachPartyPanelsFromVar();
			BC_AttachPartyFrames = nil;
		end
		-- Redraw the buttons for the target
		BeneCast_UpdateButtonsForUnit('target');

	elseif ( event == 'PLAYER_FOCUS_CHANGED' ) then
		-- Redraw the buttons for the target
		BeneCast_UpdateButtonsForUnit('focus');
	
	elseif ( event == 'UNIT_TARGET' and unit == 'target' ) then
		-- Redraw the buttons for the target of target
		BeneCast_UpdateButtonsForUnit('targettarget');
		
	elseif ( event == 'RAID_ROSTER_UPDATE' ) then
		PutDebugMsg('RAID_ROSTER_UPDATE event received');
		if BeneCastConfig.RaidSnapTo == 'STANDARDBOTTOM' or BeneCastConfig.RaidSnapTo == 'STANDARDRIGHT' then
			BC_UpdateRaid = true;
		end
				
	elseif ( event == 'PLAYER_ALIVE' or event == 'PLAYER_DEAD' or event == 'PLAYER_UNGHOST' or event == 'UPDATE_SHAPESHIFT_FORMS' ) then
		-- These changes shouldn't affect who can be shown...so we only want to update those that we've been showing before. 
		for i,attached in pairs (BC_attached_targets) do
			if attached then
				BeneCast_UpdateButtons(i);
			else
				PutDebugMsg('Not updating "'..BC_targets[i]..'" because it was not previously attached.');
			end
		end
		
	elseif ( event == 'UNIT_HEALTH' ) then
		-- Redraw a member's panel if they're dead
		if ( UnitHealth(arg1) <= 0 and UnitIsConnected(arg1) ) then
			BeneCast_UpdateButtonsForUnit(arg1);
		end
				
	elseif ( event == 'PLAYER_ENTERING_WORLD' ) then
		BeneCast_AttachPartyPanelsFromVar();
		this:RegisterEvent('ACTIONBAR_SLOT_CHANGED');
		this:RegisterEvent('SPELLS_CHANGED');
		this:RegisterEvent('SPELLCAST_FAILED');
		this:RegisterEvent('SPELLCAST_INTERRUPTED');
		this:RegisterEvent('SPELLCAST_STOP');
		this:RegisterEvent('SPELLCAST_START');
		this:RegisterEvent('PARTY_MEMBERS_CHANGED');
		this:RegisterEvent('PLAYER_TARGET_CHANGED');
		this:RegisterEvent('UNIT_AURA');
		this:RegisterEvent('PLAYER_AURAS_CHANGED');
		this:RegisterEvent('UNIT_AURASTATE');
		this:RegisterEvent('UPDATE_SHAPESHIFT_FORMS');
		this:RegisterEvent('PLAYER_ALIVE');
		this:RegisterEvent('PLAYER_DEAD');
		this:RegisterEvent('PLAYER_UNGHOST');
		this:RegisterEvent('RAID_ROSTER_UPDATE');
		this:RegisterEvent('UNIT_HEALTH');
		this:RegisterEvent('UNIT_PET');
	
	elseif ( event == 'PLAYER_LEAVING_WORLD' ) then
		this:UnregisterEvent('ACTIONBAR_SLOT_CHANGED');
		this:UnregisterEvent('SPELLS_CHANGED');
		this:UnregisterEvent('SPELLCAST_FAILED');
		this:UnregisterEvent('SPELLCAST_INTERRUPTED');
		this:UnregisterEvent('SPELLCAST_STOP');
		this:UnregisterEvent('SPELLCAST_START');
		this:UnregisterEvent('PARTY_MEMBERS_CHANGED');
		this:UnregisterEvent('PLAYER_TARGET_CHANGED');
		this:UnregisterEvent('UNIT_AURA');
		this:UnregisterEvent('PLAYER_AURAS_CHANGED');
		this:UnregisterEvent('UNIT_AURASTATE');
		this:UnregisterEvent('UPDATE_SHAPESHIFT_FORMS');
		this:UnregisterEvent('PLAYER_ALIVE');
		this:UnregisterEvent('PLAYER_DEAD');
		this:UnregisterEvent('PLAYER_UNGHOST');
		this:UnregisterEvent('RAID_ROSTER_UPDATE');
		this:UnregisterEvent('UNIT_HEALTH');
		this:UnregisterEvent('UNIT_PET');
		BeneCastProfiles[BC_CurrentProfile] = BeneCastConfig;	
	
	elseif ( event == 'PLAYER_REGEN_DISABLED' ) then
		-- Player entering combat
		BC_ConsiderInCombat = true;
		UpdateCurrentProfile();
	
	elseif ( event == 'PLAYER_REGEN_ENABLED' ) then
		-- Player exiting combat
		BC_ConsiderInCombat = false;
		UpdateCurrentProfile();
	
	end

end

function BeneCastPanelManager_OnUpdate()

	local hasMainHandEnchant, mainHandExpiration, mainHandCharges, hasOffHandEnchant, offHandExpiration, offHandCharges;
	
	hasMainHandEnchant, mainHandExpiration, mainHandCharges, hasOffHandEnchant, offHandExpiration, offHandCharges = GetWeaponEnchantInfo();
	
	if ( hasMainHandEnchant ) then
		local slotId, textureName;
	
		slotId, textureName = GetInventorySlotInfo('MainHandSlot');
	
		BeneCast_Tooltip:ClearLines();
		BeneCast_Tooltip:SetInventoryItem('player', slotId);
		
		for i = 1, BeneCast_Tooltip:NumLines() do
			if getglobal('BeneCast_TooltipTextLeft' .. i):GetText() then
				local line = getglobal('BeneCast_TooltipTextLeft' .. i):GetText();
				if ( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_ROCKBITER and string.find(line, BENECAST_STRINGS.WEAPONENCHANT_ROCKBITER) ) then
					BC_weaponenchant = BENECAST_STRINGS.WEAPONENCHANT_ROCKBITER;
					BeneCast_UpdateButtons(1);
					do break end; --WINTROW.6, break for efficiency
				end
				if ( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_FLAMETONGUE and string.find(line, BENECAST_STRINGS.WEAPONENCHANT_FLAMETONGUE) ) then
					BC_weaponenchant = BENECAST_STRINGS.WEAPONENCHANT_FLAMETONGUE;
					BeneCast_UpdateButtons(1);
					do break end; --WINTROW.6, break for efficiency
				end
				if ( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_FROSTBRAND and string.find(line, BENECAST_STRINGS.WEAPONENCHANT_FROSTBRAND) ) then
					BC_weaponenchant = BENECAST_STRINGS.WEAPONENCHANT_FROSTBRAND;
					BeneCast_UpdateButtons(1);
					do break end; --WINTROW.6, break for efficiency
				end
				if ( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_WINDFURY and string.find(line, BENECAST_STRINGS.WEAPONENCHANT_WINDFURY) ) then
					BC_weaponenchant = BENECAST_STRINGS.WEAPONENCHANT_WINDFURY;
					BeneCast_UpdateButtons(1);
					do break end; --WINTROW.6, break for efficiency
				end
				if ( BC_weaponenchant ~= BENECAST_STRINGS.WEAPONENCHANT_FEEDBACK and string.find(line, BENECAST_STRINGS.WEAPONENCHANT_FEEDBACK) ) then
					BC_weaponenchant = BENECAST_STRINGS.WEAPONENCHANT_FEEDBACK;
					BeneCast_UpdateButtons(1);
					do break end; --WINTROW.6, break for efficiency
				end
			end
		end
	else
		if ( BC_weaponenchant ) then
			BC_weaponenchant = nil;
			BeneCast_UpdateButtons(1);
		end
		BC_weaponenchant = nil;
	end

	-- Update all button-flashing at once
	local buffAlphaValue = nil;
	for i,buttonname in pairs (BC_buttons_that_flash) do
		local button = getglobal(buttonname);
		if button then
			if buffAlphaValue == nil then --this way it only has to be calculated once
				if ( BuffFrameFlashState == 1 ) then
					buffAlphaValue = (BUFF_FLASH_TIME_ON - BuffFrameFlashTime) / BUFF_FLASH_TIME_ON;
					buffAlphaValue = buffAlphaValue * (1 - BUFF_MIN_ALPHA) + BUFF_MIN_ALPHA;
				else
					buffAlphaValue = BuffFrameFlashTime / BUFF_FLASH_TIME_ON;
					buffAlphaValue = (buffAlphaValue * (1 - BUFF_MIN_ALPHA)) + BUFF_MIN_ALPHA;
					--this:SetAlpha(BuffFrameFlashTime / BUFF_FLASH_TIME_ON);
				end
			end
			button:SetAlpha(buffAlphaValue);
		end
	end
	
	if BeneCastConfig.ShowRaid then
		local docheck = false;
		if BC_UpdateRaid then
			if BC_Last_UpdateRaid_Done == nil then
				docheck = true;
			elseif (GetTime() - BC_Last_UpdateRaid_Done) >= 5 then
				docheck = true;
			end
		end
		if docheck then
			BC_Last_UpdateRaid_Done = GetTime();
			BC_UpdateRaid = nil;
			BeneCast_UpdateRaid();
		end
	end
		
	-- Check for buttons if they are within range
	if BeneCastConfig.CheckRange then
		local docheck = false;
		if ( BC_LastRangeCheck == nil ) or
		   ( (GetTime() - BC_LastRangeCheck) > 1 ) then -- Every 1 second
		   	docheck = true;
		end
		if docheck then
			BC_LastRangeCheck = GetTime();
			BeneCast_CheckRange();
		end
	end

end

function BeneCast_CheckRange()

	for id = 2, table.getn(BC_targets) do
		local unit = BeneCast_GetUnitIDFromPanelID(id);
		if getglobal('BeneCastPanel' .. id):IsVisible() and UnitExists(unit) then
			for i = 1, BENECAST_MAX_NUMBER_OF_BUTTONS do
				local buttonname = 'BeneCastPanel' .. id .. 'Button' .. i;
				local button = getglobal(buttonname);
				if button:IsVisible() then
					local spelltype = button.spelltype;
					local i = BC_spell_data[spelltype].spellcount
					local spellname = BC_spell_data[spelltype]['spells'][i].name;
					if BC_spell_data[spelltype]['spells'][i].rank then
						spellname = spellname .. '(' .. BC_spell_data[spelltype]['spells'][i].rank .. ')';
					else
						spellname = spellname .. '()';
					end
					if SpellHasRange(spellname) then
						if not IsSpellInRange(spellname,unit) then
							getglobal(buttonname .. 'Range'):Show();
							if unit == 'target' then
								PutDebugMsg('Button ' .. i .. ', spell ' .. spellname .. ' out of range');
							end
						else
							getglobal(buttonname .. 'Range'):Hide();
						end
					else
						getglobal(buttonname .. 'Range'):Hide();
					end
				end
			end
		end
	end

end

-- *****************************************************************************
-- BeneCastResetButton functions
-- *****************************************************************************

function BeneCast_ResetTooltip()

	GameTooltip:SetOwner(this, 'ANCHOR_RIGHT');
	GameTooltip:SetText(BENECAST_STRINGS['TEXT_RESET_BUTTON_TOOLTIP'], nil, nil, nil, nil, 1);

end

function BeneCast_ResetProfiles()

	--Reset the BeneCastProfiles-table
	BeneCastProfiles = {};
	
	BeneCast_ResetConfig();
	
	BeneCastProfiles['<Default>'] = BeneCastConfig;
	if not BeneCastSetup then
		BeneCastSetup = {};
	end
	BeneCastSetup['PrefferedInCombatProfile'] = '<Default>';
	BeneCastSetup['PrefferedOutOfCombatProfile'] = '<Default>';
	
end

function BeneCast_ResetConfig()

	-- Reset the BeneCastConfig table
	BeneCastConfig = {};
	
	BeneCastConfig[1] = {};
	BeneCastConfig[2] = {};
	BeneCastConfig[3] = {};
	BeneCastConfig[4] = {};
	BeneCastConfig[5] = {};
	BeneCastConfig[6] = {};
	BeneCastConfig[7] = {};
	BeneCastConfig[8] = {};
	BeneCastConfig[9] = {};
	BeneCastConfig[10] = {};

	for i = 1,10 do
		for type in pairs (BeneCast_SpellLevel[BC_class]) do
			BeneCastConfig[i][type] = false;
		end
	end
	
	BeneCastConfig['DmgBasedHeal'] = true;
	BeneCastConfig['ManaBasedHeal'] = true;
	BeneCastConfig['ReverseHoTs'] = false; 
	BeneCastConfig['Overheal'] = false;
	BeneCastConfig['ShowAllButtons'] = false;
	BeneCastConfig['FlashAsFade'] = false;
	BeneCastConfig['HideMinimap'] = false;
	BeneCastConfig['ShowTooltips'] = true;
	BeneCastConfig['ShowTooltipsName'] = false;
	BeneCastConfig['Unlock'] = false;
	BeneCastConfig['Notification'] = false;
	BeneCastConfig['SelfNotify'] = true;
	BeneCastConfig['PartyNotify'] = false;
	BeneCastConfig['SayNotify'] = false;
	BeneCastConfig['RaidNotify'] = false;
	BeneCastConfig['TargetNotify'] = false;
	BeneCastConfig['UserNotify'] = false;
	BeneCastConfig['SelfCasts'] = true;
	BeneCastConfig['NotifyRank'] = true;
	BeneCastConfig['NotifyTime'] = true;
	BeneCastConfig['NotifyHeal'] = true;
	BeneCastConfig['NotifyMaxHeal'] = false;
	BeneCastConfig['NotifyCure'] = true;
	BeneCastConfig['NotifyBuff'] = true;
	BeneCastConfig['NotifyRes'] = true;
	BeneCastConfig['PlayerAsDefault'] = true;
	BeneCastConfig['ShowPets'] = false;
	BeneCastConfig['ShowRaid'] = false;
	BeneCastConfig['ShowRaidPets'] = false; 
	BeneCastConfig['CheckRange'] = false; 
	BeneCastConfig['ShowTargetTarget'] = true; 
	
	BeneCastConfig['ButtonSize'] = 36;
	BeneCastConfig['OverhealSlider'] = 1;
	BeneCastConfig['ButtonNumber'] = 5;
	
	BeneCastConfig['UserChannel'] = nil;
	
	BeneCastConfig['SnapTo'] = 'STANDARD';
	BeneCastConfig['RaidSnapTo'] = 'STANDARDRIGHT';
	
	BeneCastConfig['Debug'] = false;
	
	BeneCast_ResetData();
	BeneCast_LoadOptions();
	-- Place the panels in the correct place
	BeneCast_SnapPanels(BeneCastConfig.SnapTo);
	BeneCast_AttachPartyPanelsFromVar();
	BeneCastConfig['Version'] = BENECAST_VERSION;
	
	-- Reset the SnapTo selections (Raid too, thnx Shimavak)
	UIDropDownMenu_SetSelectedValue(BeneCastSnapToMenu, BeneCastConfig.SnapTo);
	UIDropDownMenu_SetSelectedValue(BeneCastRaidSnapToMenu, BeneCastConfig.RaidSnapTo);
	-- Redraw everyone's panels
	for i in pairs (BC_targets) do
		BeneCast_OrientButtons(i);
		BeneCast_UpdateButtons(i);
	end

end

-- *****************************************************************************
-- BeneCastUserChannel functions
-- *****************************************************************************

function BeneCastUserChannel_OnTextChange()

	BeneCastConfig['UserChannel'] = this:GetText();

end

-- *****************************************************************************
-- BeneCastEditBox functions
-- *****************************************************************************

function BeneCastEditBox_ApplyChange()
	
	if this.var1 == 'variable' then
		if this.var2 == 'BC_PanelNumber' then
			BC_PanelNumber = this:GetText();
			BeneCastUpdateCustomEditBoxes();
		end
		
	elseif this.var1 == 'panelinfo' then
		local framename = 'BeneCastPanel' .. BC_PanelNumber;
		if BeneCastConfig.Debug then
			local msg = 'Updating BeneCastConfig[';
			if framename then
				msg = msg .. framename .. '][';
			else
				msg = msg .. 'nil][';
			end
			if this.var2 then
				msg = msg .. this.var2 .. ']';
			else
				msg = msg .. 'nil]';
			end
			PutDebugMsg(msg);
		end
		local boxvalue = this:GetText();
		if boxvalue ~= BeneCastConfig[framename][this.var2] then
			BeneCastConfig['SnapTo'] = 'CUSTOM';
			UIDropDownMenu_SetSelectedValue(BeneCastSnapToMenu, 'CUSTOM');
			BeneCastConfig[framename][this.var2] = boxvalue;
			BeneCastConfig[framename]['Moved'] = false;
		end
		PutDebugMsg('Updating Panel ' .. BC_PanelNumber);
		BeneCast_AttachPanelsFromVar(tonumber(BC_PanelNumber));
		
	elseif this.var1 == 'profilename' then
		local newtext = this.GetText();
		if nextext == '' then
			newtext = nil;
		end
		if newtext ~= this.oldtext then
			-- Change name of existing profile
			if this.oldtext and newtext then
				BeneCastProfiles[newtext] = BeneCastProfiles[this.oldtext];
				BeneCastProfiles[this.oldtext] = nil;
				if BeneCastSetup['PrefferedInCombatProfile'] == this.oldtext then
					BeneCastSetup['PrefferedInCombatProfile'] = newtext;
				end
				if BeneCastSetup['PrefferedOutOfCombatProfile'] == this.oldtext then
					BeneCastSetup['PrefferedOutOfCombatProfile'] = newtext;
				end
				if BC_CurrentProfile == this.oldtext then
					BC_CurrentProfile = newtext;
				end
			-- Create new profile
			elseif not this.oldtext and newtext then
				BeneCastProfiles[newtext] = BeneCastConfig;
			-- Delete existing profile
			elseif this.oldtext and not this:GetText() then
				BeneCastProfiles[this.oldtext] = nil;
			end
			BeneCast_UpdateProfileCheckButtons();
		end
		
	end
	
end

-- *****************************************************************************
-- BeneCastSnapToMenu functions
-- *****************************************************************************

function BeneCastSnapToMenu_Initialize()
	
	local info;
	-- Add SnapTo options
	for snapto in pairs (BeneCast_SnapTo) do
		if IsAddOnLoaded(BeneCast_SnapTo[snapto].AddOn) or snapto == 'CUSTOM' then
			info = {};
			if ( BeneCast_SnapTo[snapto].AddOn == 'BeneCast' ) then
				info.text = BENECAST_STRINGS.TEXT_STANDARD;
			elseif ( BeneCast_SnapTo[snapto].AddOn == 'Perl_Target' ) then
				info.text = BENECAST_STRINGS.TEXT_PERLCLASSIC;
			elseif ( snapto == 'CUSTOM' ) then
				info.text = 'Custom';
			else
				info.text = BeneCast_SnapTo[snapto].AddOn;
			end
			
			info.value = snapto;
			if ( BeneCastConfig.SnapTo == snapto ) then
				info.checked = true;
			else
				info.checked = false;
			end
			info.func = BeneCastSnapToMenu_SetSnap;
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);
		end
	end

end

function SetBeneCastSnapTo(snapto_option)
	local found;
	for snaptoval in pairs (BeneCast_SnapTo) do
		if BeneCast_SnapTo[snaptoval].AddOn == snapto_option then
			found = true;
			BeneCastConfig.SnapTo = snaptoval;
			BeneCast_SnapPanels(snapto_option);
			BeneCast_AttachPartyPanelsFromVar(); --WINTROW.6
			UIDropDownMenu_SetSelectedValue(BeneCastSnapToMenu, snaptoval);
			for i in pairs (BC_targets) do
				BeneCast_UpdateButtons(i);
			end
			do break end;
		end
	end
	if not found then
		PutDebugMsg(snapto_option .. ' not found in predefined unit frames');
	end
end

function SetBeneCastSnapTo2(snapto_option)
	if ( not BeneCast_SnapTo[snapto_option] ) then
		PutDebugMsg(snapto_option .. ' not found in predefined unit frames');
		return;
	end
	BeneCastConfig.SnapTo = snapto_option;
	BeneCast_SnapPanels(BeneCast_SnapTo[snapto_option].AddOn);
	BeneCast_AttachPartyPanelsFromVar(); --WINTROW.6
	UIDropDownMenu_SetSelectedValue(BeneCastSnapToMenu, snapto_option);
	for i in pairs (BC_targets) do
		BeneCast_UpdateButtons(i);
	end
end

function PutDebugMsg(msg)
	if ( BeneCastConfig['Debug'] and DEFAULT_CHAT_FRAME ) then
		DEFAULT_CHAT_FRAME:AddMessage('Debug: ' .. msg);
	end
end

function PutMsg(msg)
	if ( DEFAULT_CHAT_FRAME ) then
		DEFAULT_CHAT_FRAME:AddMessage(msg);
	end
end

function BeneCastSnapToMenu_SetSnap()

	if ( not this.value ) then
		return;
	end
	PutDebugMsg('Entered BeneCastSnapToMenu_SetSnap'); --WINTROW.6
	BeneCastConfig.SnapTo = this.value;
	PutDebugMsg('SnapTo = ' .. this.value); --WINTROW.6
	BeneCast_SnapPanels(BeneCast_SnapTo[this.value].AddOn);
	BeneCast_AttachPartyPanelsFromVar(); --WINTROW.6
	UIDropDownMenu_SetSelectedValue(BeneCastSnapToMenu, this.value);
	for i in pairs (BC_targets) do
		BeneCast_UpdateButtons(i);
	end
	PutDebugMsg('Updated buttons'); --WINTROW.6

end

function BeneCastSnapToMenu_Load()

	UIDropDownMenu_Initialize(this, BeneCastSnapToMenu_Initialize);
	UIDropDownMenu_SetSelectedValue(this, BeneCastConfig.SnapTo);
	getglobal('BeneCastSnapToMenuLabel'):SetText(BENECAST_STRINGS.TEXT_SNAP_TO);
	BeneCastSnapToMenu.tooltip = BENECAST_STRINGS.TEXT_SNAP_TO_TOOLTIP;
	UIDropDownMenu_SetWidth(110, BeneCastSnapToMenu);

end

-- *****************************************************************************
-- BeneCastRaidSnapToMenu functions
-- *****************************************************************************

function BeneCastRaidSnapToMenu_Initialize()
	
	local info;
	-- Add SnapTo options
	for snapto in pairs (BeneCast_RaidSnapTo) do
		if ( IsAddOnLoaded(BeneCast_RaidSnapTo[snapto].AddOn) ) then
			info = {};
			if ( BeneCast_RaidSnapTo[snapto].AddOn == 'BeneCast' ) then
				info.text = BENECAST_STRINGS.TEXT_STANDARD;
			else
				info.text = BeneCast_RaidSnapTo[snapto].AddOn;
			end
			-- Determine the position for BeneCast panels to snap to
			if ( string.find(snapto, 'BOTTOM') ) then
				info.text = info.text .. BENECAST_STRINGS.TEXT_BOTTOM;
			elseif ( string.find(snapto, 'LEFT') ) then
				info.text = info.text .. BENECAST_STRINGS.TEXT_LEFT;
			elseif ( string.find(snapto, 'RIGHT') ) then
				info.text = info.text .. BENECAST_STRINGS.TEXT_RIGHT;
			elseif ( string.find(snapto, 'TOP') ) then
				info.text = info.text .. BENECAST_STRINGS.TEXT_TOP;
			end
			
			info.value = snapto;
			if ( BeneCastConfig.RaidSnapTo == snapto ) then
				info.checked = true;
			else
				info.checked = false;
			end
			info.func = BeneCastRaidSnapToMenu_SetSnap;
			UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);
		end
	end

end

function BeneCastRaidSnapToMenu_SetSnap()

	BeneCastConfig.RaidSnapTo = this.value;
	UIDropDownMenu_SetSelectedValue(BeneCastRaidSnapToMenu, this.value);
	BC_UpdateRaid = true;
	
end

function BeneCastRaidSnapToMenu_Load()

	UIDropDownMenu_Initialize(this, BeneCastRaidSnapToMenu_Initialize);
	UIDropDownMenu_SetSelectedValue(this, BeneCastConfig.RaidSnapTo);
	getglobal('BeneCastRaidSnapToMenuLabel'):SetText(BENECAST_STRINGS.TEXT_RAID_SNAP_TO);
	BeneCastRaidSnapToMenu.tooltip = BENECAST_STRINGS.TEXT_RAID_SNAP_TO_TOOLTIP;

end

-- *****************************************************************************
-- BeneCastNotificationDropDown functions
-- *****************************************************************************

function BeneCastNotificationDropDown_Initialize()
	
	local info, buttonIndex;

	-- Add the menu title which will be the default selected item.
	info = {};
	info.text = BENECAST_STRINGS.TEXT_USER_CHANNEL;
	info.value = 'Tip';
	info.isTitle = true;
	info.checked = false;

	UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);

	-- Finds the index of the associated dropdown
	buttonIndex = this:GetID();
	if buttonIndex == 0 then
		buttonIndex = this:GetParent():GetID();
	end
	
	-- Finds the config of the current dropdown box
	local values = BeneCastOptionDropDowns[buttonIndex];
	if values == nil then
		PutDebugMsg('no BeneCastOptionDropDowns for id ' .. buttonIndex .. ' object ' .. this:GetName());
	end

	-- Add Available notification targets
	for textReference, channel in pairs (BeneCastNotificationTargets) do
		info = {};
		info.text = BENECAST_STRINGS[textReference];
		--info.value = { chan = channel,
		--               cvar = values.cvar,
		--	       dropdownname = 'BeneCastOptionDropDown' .. buttonIndex };
		info.value = channel;
		info.func = BeneCastNotificationDropDown_OnClick;
		info.keepShownOnClick = true;
		--info.tooltipTitle = BENECAST_STRINGS[textReference .. '_TOOLTIP'];
		--info.tooltipText = BENECAST_STRINGS[textReference];
		if BeneCastConfig[values.cvar] == channel then
			info.checked = true;
		end
		
		UIDropDownMenu_AddButton(info, UIDROPDOWNMENU_MENU_LEVEL);
	end

end

function BeneCastNotificationDropDown_OnClick()

	-- Finds the index of the associated dropdown
	local buttonIndex = this:GetParent():GetID();
	PutDebugMsg('Clicked on ' .. this:GetName() .. ' id=' .. buttonIndex);
	local cvar;
	if BeneCastOptionDropDowns[buttonIndex] then
		cvar = BeneCastOptionDropDowns[buttonIndex].cvar;
		BeneCastConfig[cvar] = this.value;
		UIDropDownMenu_SetSelectedValue(getglobal('BeneCastOptionDropDown' .. buttonIndex), this.value);
		if BeneCastConfig.CascadeChannels then
			for i,values in pairs (BeneCastOptionDropDowns) do
				BeneCastConfig[values.cvar] = this.value;
				getglobal('BeneCastOptionDropDown' .. i):Show();
			end
		end
	end
	
end

function BeneCastNotificationDropDown_Load()

	-- Finds the index of the associated dropdown
	local buttonIndex = this:GetID();
	if buttonIndex == 0 then
		buttonIndex = this:GetParent():GetID();
	end
	local cvar = BeneCastOptionDropDowns[buttonIndex].cvar;
	UIDropDownMenu_Initialize(this, BeneCastNotificationDropDown_Initialize);
	if BeneCastConfig[cvar] then
		UIDropDownMenu_SetSelectedValue(this,BeneCastConfig[cvar]);
	else
		UIDropDownMenu_SetSelectedValue(this, 'Tip');
	end
end

-- *****************************************************************************
-- BeneCastOptionFrame functions
-- *****************************************************************************

function BeneCastOptionFrame_OnLoad()
	
	-- Binding labels
	BINDING_HEADER_BENECAST = 'BeneCast';
	BINDING_NAME_BENECAST_OPTIONS = BENECAST_STRINGS.TEXT_BENECAST_OPTIONS;
	BINDING_NAME_BENECAST_TOGGLE_BUFFS = BENECAST_STRINGS.TEXT_BENECAST_TOGGLE_BUFFS;
	BINDING_NAME_BENECAST_BUTTON1 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON1;
	BINDING_NAME_BENECAST_BUTTON2 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON2;
	BINDING_NAME_BENECAST_BUTTON3 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON3;
	BINDING_NAME_BENECAST_BUTTON4 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON4;
	BINDING_NAME_BENECAST_BUTTON5 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON5;
	BINDING_NAME_BENECAST_BUTTON6 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON6;
	BINDING_NAME_BENECAST_BUTTON7 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON7;
	BINDING_NAME_BENECAST_BUTTON8 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON8;
	BINDING_NAME_BENECAST_BUTTON9 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON9;
	BINDING_NAME_BENECAST_BUTTON10 = BENECAST_STRINGS.TEXT_BENECAST_BUTTON10;
	SLASH_BENECAST1 = '/benecast';
	SLASH_BENECAST2 = '/bc';
	
	-- Setup the BeneCast Slash Command Handler
	SlashCmdList['BENECAST'] = function(msg)
		BeneCast_SlashCommandHandler(msg);
	end
	
	-- Set up the title in the option frame
	getglobal('BeneCastTitleText'):SetText(BENECAST_STRINGS.TEXT_TITLE);

	-- Check Tab 1 and show the Player spell selection subframe
	getglobal('BeneCastPlayerFrame'):Show();
	getglobal('BeneCastOptionFrameTab1'):SetChecked(1);
	getglobal('BeneCastOptionFrameTab1'):Disable();
	
	-- Setup Tab Text and Graphics
	getglobal('BeneCastOptionFrameTab1NormalTexture'):SetTexture('Interface\\Icons\\Spell_Nature_WispSplode');
	getglobal('BeneCastOptionFrameTab1').tooltipText = UnitName('player');
	
	getglobal('BeneCastOptionFrameTab2NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab2NormalTexture'):SetTexCoord(0.7421875, 0.98828125, 0, 0.25);
	getglobal('BeneCastOptionFrameTab2').tooltipText = BENECAST_STRINGS.CLASS_DRUID;
	
	getglobal('BeneCastOptionFrameTab3NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab3NormalTexture'):SetTexCoord(0, 0.25, 0.25, 0.5);
	getglobal('BeneCastOptionFrameTab3').tooltipText = BENECAST_STRINGS.CLASS_HUNTER;
	
	getglobal('BeneCastOptionFrameTab4NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab4NormalTexture'):SetTexCoord(0.25, 0.49609375, 0, 0.25);
	getglobal('BeneCastOptionFrameTab4').tooltipText = BENECAST_STRINGS.CLASS_MAGE;
	
	getglobal('BeneCastOptionFrameTab5NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab5NormalTexture'):SetTexCoord(0.49609375, 0.7421875, 0.25, 0.5);
	getglobal('BeneCastOptionFrameTab5').tooltipText = BENECAST_STRINGS.CLASS_PRIEST;
	
	getglobal('BeneCastOptionFrameTab6NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab6NormalTexture'):SetTexCoord(0.49609375, 0.7421875, 0, 0.25);
	getglobal('BeneCastOptionFrameTab6').tooltipText = BENECAST_STRINGS.CLASS_ROGUE;
	
	getglobal('BeneCastOptionFrameTab7NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab7NormalTexture'):SetTexCoord(0.7421875, 0.98828125, 0.25, 0.5);
	getglobal('BeneCastOptionFrameTab7').tooltipText = BENECAST_STRINGS.CLASS_WARLOCK;
	
	getglobal('BeneCastOptionFrameTab8NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab8NormalTexture'):SetTexCoord(0, 0.25, 0, 0.25);
	getglobal('BeneCastOptionFrameTab8').tooltipText = BENECAST_STRINGS.CLASS_WARRIOR;

	getglobal('BeneCastOptionFrameTab9NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab9NormalTexture'):SetTexCoord(0.25, 0.49609375, 0.25, 0.5);
	getglobal('BeneCastOptionFrameTab9').tooltipText = BENECAST_STRINGS.CLASS_SHAMAN;
	
	getglobal('BeneCastOptionFrameTab10NormalTexture'):SetTexture('Interface\\Glues\\CharacterCreate\\UI-CharacterCreate-Classes');
	getglobal('BeneCastOptionFrameTab10NormalTexture'):SetTexCoord(0.0, 0.25, 0.5, 0.75);
	getglobal('BeneCastOptionFrameTab10').tooltipText = BENECAST_STRINGS.CLASS_PALADIN;
	
	getglobal('BeneCastOptionFrameTab11NormalTexture'):SetTexture('Interface\\Icons\\INV_Misc_Wrench_01');
	getglobal('BeneCastOptionFrameTab11').tooltipText = BENECAST_STRINGS.TEXT_SETUP;
	
	getglobal('BeneCastOptionFrameTab12NormalTexture'):SetTexture('Interface\\Icons\\INV_Letter_08');
	getglobal('BeneCastOptionFrameTab12').tooltipText = BENECAST_STRINGS.TEXT_NOTIFICATION;
	
	getglobal('BeneCastOptionFrameTab13NormalTexture'):SetTexture('Interface\\Icons\\INV_Misc_Head_Dragon_01');
	getglobal('BeneCastOptionFrameTab13').tooltipText = BENECAST_STRINGS.TEXT_RAID;

	getglobal('BeneCastOptionFrameTab14NormalTexture'):SetTexture('Interface\\Icons\\INV_Misc_Wrench_02');
	getglobal('BeneCastOptionFrameTab14').tooltipText = 'Unit Frames';

end

-- *****************************************************************************
-- BeneCastMiniMapButton Game Event Handlers
-- *****************************************************************************

-- Clicks on the BeneCast button do different things
function BeneCastOptionFrameToggle()

	-- If the mouseButton is the left button toggle BeneCastFrame's visibility
	-- if ( arg1 == 'LeftButton' ) then
		local frame = getglobal('BeneCastOptionFrame');
		if ( frame:IsVisible() ) then
			frame:Hide();
		else
			frame:Show();
		end
	--end
	
end

function BeneCastMiniMap_OnMouseDown()

	if ( ( BeneCastConfig.Unlock ) and ( arg1 == 'LeftButton' ) ) then
		this:StartMoving();
		this.isMoving = true;
	end

end

function BeneCastMiniMap_OnLoad()

	this:RegisterEvent('ADDON_LOADED');

end

function BeneCastMiniMap_OnEvent()

	if ( event == 'ADDON_LOADED' and arg1 == 'BeneCast' ) then
		if ( BeneCastConfig.HideMinimap ) then
			getglobal('BeneCastMinimapButton'):Hide();
		end
	end

end
