
--[ Mod data ]--
VisualHeal = {};
VisualHeal.Data = 
{
    Name = "VisualHeal",
    Version = 7,
    CommVersion = 1
};

VisualHeal.ClassColours =
{
    PALADIN = "|cFFF48CBA",
    WARRIOR = "|cFFC69B6D",
    WARLOCK = "|cFF9382C9",
    PRIEST  = "|cFFFFFFFF",
    DRUID   = "|cFFFF7C0A",
    MAGE    = "|cFF68CCEF",
    ROGUE   = "|cFFFFF468",
    SHAMAN  = "|cFFF48CBA",
    HUNTER  = "|cFFAAD372"
}

VisualHeal.Healers = {};
VisualHeal.Healing = {};
VisualHeal.Monitor = {};
VisualHeal.SpellCache = {};

--[ Settings ]--
VisualHealOptions = {};
local DefaultOptions = -- Default values for options
{
    ShowHealBar = true
}

--[ Utilities ]--

-- Write one line to the chat frame
function VisualHeal:Print(text, red, green, blue)
   if DEFAULT_CHAT_FRAME then
       DEFAULT_CHAT_FRAME:AddMessage(text, red or 1.0, green or 1.0, blue or 1.0);
   end
end

function VisualHeal:MultiPrint(...)
    if DEFAULT_CHAT_FRAME then
        local i;   
        local msg = '';
        for i=1, select("#", ...) do 
            msg = msg .. tostring(select(i, ...)) .. ' : ';
        end
        self:Print(msg);
   end
end

-- Returns true if the player is in a raid group
function VisualHeal:InRaid()
    return (GetNumRaidMembers() > 0);
end

-- Returns true if the player is in a party or a raid
function VisualHeal:InParty()
    return (GetNumPartyMembers() > 0);
end

-- Returns true if health information is available for the unit
function VisualHeal:UnitHasHealthInfo(unit)
    if (not unit) then 
        return false; 
    end
    if (UnitIsUnit('player', unit) or UnitIsUnit('pet', unit)) then 
        return true;
    end
    if (self:InRaid()) then
        -- In raid
        if (UnitPlayerOrPetInRaid(unit)) then 
            return true;
        end
    else
        -- Not in raid
        if (UnitPlayerOrPetInParty(unit)) then 
            return true;
        end
    end
    return false;
end

-- Determine if player is in a battleground
function VisualHeal:InBattleground()
    local id;
    for i=1,MAX_BATTLEFIELD_QUEUES do
        _, _, id = GetBattlefieldStatus(i);
        if id ~= 0 then return true end
    end
    return false;
end

-- Convert name to unit ID from a subset of all possible solutions
function VisualHeal:UnitNameToUnitID(name)
    local i;
    local id;
    if (name == UnitName('player')) then
        return 'player';
    end
    if (name == UnitName('pet')) then
        return 'pet';
    end
    if (self:InRaid()) then
        for i=1,GetNumRaidMembers() do
            id = 'raid' .. i;
            if (name == UnitName(id)) then
                return id;
            end
        end
        for i=1,GetNumRaidMembers() do
            id = 'raidpet' .. i;
            if (name == UnitName(id)) then
                return id;
            end
        end
    elseif self:InParty() then
        for i=1,GetNumPartyMembers() do
            id = 'party' .. i;
            if (name == UnitName(id)) then
                return id;
            end
        end
        for i=1,GetNumPartyMembers() do
            id = 'partypet' .. i;
            if (name == UnitName(id)) then
                return id;
            end
        end
    end
    if (name == UnitName('target')) then
        return 'target';
    end
    if (name == UnitName('targettarget')) then
        return 'targettarget';
    end
end

function VisualHeal:EstimateUnitHealth(unit)
    local _, class = UnitClass(unit);
    class = class or "Unknown";
    local maxHealthTab = {
    warrior=4100,
    paladin=4000,
    shaman=3500,
    rogue=3100,
    hunter=3100,
    druid=3100,
    warlock=2300,
    mage=2200,
    priest=2100};
    return ((UnitHealth(unit) or 0) * (maxHealthTab[string.lower(class)] or 4000) * (UnitLevel(unit) or 60) / 3600);
end

function VisualHeal:GetSpellBonusHealingPenalty(spellLevel, playerLevel)
    local num = spellLevel + 6;

    if (num > playerLevel) then
        return 1;
    else
        return num / playerLevel;
    end
end

-- Detects if a buff is present on the unit and returns the application number
function VisualHeal:DetectBuff(unit, name, app)
    local i=1;
    local state, apps;
    while true do
        _, _, state, apps = UnitBuff(unit, i);
        if (not state) then 
            return false;
        end
        if (string.find(state, name) and ((app == apps) or (app == nil))) then 
            return apps;
        end
        i=i+1;
    end
end

-- Detects if a debuff is present on the unit and returns the application number
function VisualHeal:DetectDebuff(unit, name, app)
    local i=1;
    local state, apps;
    while true do
        _, _, state, apps = UnitDebuff(unit, i);
        if (not state) then 
            return false;
        end
        if (string.find(state,name) and ((app == apps) or (app == nil))) then 
            return apps;
        end
        i=i+1;
    end
end

-- List buffs and debuffs on a unit
function VisualHeal:ListUnitEffects(unit)
    if UnitExists(unit) then
        local i=1;
        self:Print("|cffffff80******* Buffs on " .. (UnitName(unit) or "Unknown") .. " *******|r");
        while (UnitBuff(unit,i)) do
            local string;
            VisualHeal_ScanningTooltip:ClearLines();
            VisualHeal_ScanningTooltip:SetUnitBuff(unit,i);
            local icon,apps = UnitBuff(unit,i);
            string = "|cff0080ff" .. (VisualHeal_ScanningTooltipTextLeft1:GetText() or "") .. ":|r|cffffd200 ";
            string = string .. (VisualHeal_ScanningTooltipTextRight1:GetText() or "") .. ", ";
            string = string .. icon .. ", ";
            string = string .. apps .. "|r\n";
            string = string .. ">" .. (VisualHeal_ScanningTooltipTextLeft2:GetText() or "");
            self:Print(string);
            i=i+1;
        end
        i=1;
        self:Print("|cffffff80******* DeBuffs on " .. (UnitName(unit) or "Unknown") .. " *******|r");
        while (UnitDebuff(unit,i)) do
            local string;
            VisualHeal_ScanningTooltip:ClearLines();
            VisualHeal_ScanningTooltip:SetUnitDebuff(unit,i);
            local icon,apps = UnitDebuff(unit,i);
            string = "|cff0080ff" .. (VisualHeal_ScanningTooltipTextLeft1:GetText() or "") .. ":|r|cffffd200 ";
            string = string .. (VisualHeal_ScanningTooltipTextRight1:GetText() or "") .. ", ";
            string = string .. icon .. ", ";
            string = string .. apps .. "|r\n";
            string = string .. ">" .. (VisualHeal_ScanningTooltipTextLeft2:GetText() or "");
            self:Print(string);
            i=i+1;
        end
    end
end

function VisualHeal:SetDefaultParameters()
    for k in pairs(DefaultOptions) do
        VisualHealOptions[k] = DefaultOptions[k];
    end
end

-- SpellCache[spell][rank][stat]
-- stat: Mana, Heal
function VisualHeal:GetSpellInfo(name)

    -- Check if info is already cached
    if (self.SpellCache[name]) then
        return self.SpellCache[name];
    end

    self.SpellCache[name] = {};

    -- Gather info (only done if not in cache)
    local i = 1;

    while true do

        local spellName, spellRank = GetSpellName(i, BOOKTYPE_SPELL);
        
        if (not spellName) then 
            break 
        end

        if (spellName == name) then
            -- This is the spell we're looking for, gather info

            -- Determine rank
            _, _, spellRank = string.find(spellRank, " (%d+)$");
            spellRank = tonumber(spellRank);
            VisualHeal_ScanningTooltip:ClearLines();
            VisualHeal_ScanningTooltip:SetSpell(i, BOOKTYPE_SPELL);
    
            -- Determine mana
            local _, _, Mana = string.find(VisualHeal_ScanningTooltipTextLeft2:GetText() or "","^(%d+)");
            Mana = tonumber(Mana);
            if not (type(Mana) == "number") then Mana = 0 end

            -- Determine healing
            local _, _, HealMin, HealMax = string.find(VisualHeal_ScanningTooltipTextLeft4:GetText() or "", VISUALHEAL_PATTERN_NUMBERRANGE);
            HealMin,HealMax = tonumber(HealMin),tonumber(HealMax);
            local Heal = (HealMin+HealMax)/2;

            self.SpellCache[spellName][spellRank] = {Mana = Mana, Heal = Heal};
        end
        i = i + 1;
    end
    return self.SpellCache[name];
end

VisualHeal.RelicSlotNumber = GetInventorySlotInfo("RangedSlot");
function VisualHeal:GetEquippedRelicID()
    local itemLink = GetInventoryItemLink('player', self.RelicSlotNumber);
    if (itemLink) then
        local _, _, itemID = strfind(itemLink, "(%d+):");
        return tonumber(itemID);
    end
end

--[ Healing Database Functions ]--

function VisualHeal:EntryUpdate(healerName, targetName, healSize, healTime)

    if (targetName) then
        -- Register healer
        self.Healers[healerName] = targetName;
    else
        -- If unspecified determine the targetName
        targetName = self.Healers[healerName];
    end

    if (not targetName) then
        return;
    end

    if (self.Debug) then
        self:MultiPrint("EntryUpdate", healerName or "nil", targetName or "nil", healSize or "nil", healTime or "nil");
    end

    -- Create entry for targetName if it does not exist
    if (not self.Healing[targetName]) then 
        self.Healing[targetName] = {}; 
    end

    if (self.Healing[targetName][healerName]) then
        -- This is an update of earlier info
        if (healTime) then
            self.Healing[targetName][healerName].Time = healTime;
        end
        if (healSize) then
            self.Healing[targetName][healerName].Size = healSize or self.Healing[targetName][healerName].Size;
        end
    else
        -- This is new info
        self.Healing[targetName][healerName] = {Time = healTime, Size = healSize};
    end
end

function VisualHeal:EntryDelete(healerName)
    local targetName = self.Healers[healerName];
    if (targetName) then
        self.Healers[healerName] = nil;
        self.Healing[targetName][healerName].Time = nil;
        self.Healing[targetName][healerName].Size = nil;
        if (self.Debug) then
            self:MultiPrint("EntryDelete", healerName, targetName);
        end
    end
end

function VisualHeal:GetIncommingHeal(targetName, endTime)
    if self.Healing[targetName] then
        local incommingHeal = 0;
        for i,v in pairs(self.Healing[targetName]) do
            if (v.Size and v.Time) then
                if (v.Time < GetTime()) then
                    v.Time = nil;
                    v.Size = nil;
                elseif (v.Time < endTime)  then
                    incommingHeal = incommingHeal + v.Size;
                end
            end
            return incommingHeal;
        end
    else
        return 0;
    end
end

function VisualHeal:CommSend()
    -- VisualHeal CommVersion:HealingTargetName:HealSize:TimeLeft
    SendAddonMessage("VisualHeal", self.Data.CommVersion .. ":" .. self.Monitor.HealingTargetName .. ":" .. math.floor(self.Monitor.Size) .. ":" .. (math.floor(1000 * (self.Monitor.Time - GetTime())) / 1000), self:InBattleground() and "BATTLEGROUND" or "RAID");
end

--[ Healing Bar Functions ]--

function VisualHeal:UpdateHealBar()
    local hp, hppre, hppost, waste;
    local healingTargetUnitID = self.Monitor.HealingTargetUnitID;
    local incommingHeal = self:GetIncommingHeal(self.Monitor.HealingTargetName, self.Monitor.Time);

    -- Determine health percentages of healing target
    if self:UnitHasHealthInfo(healingTargetUnitID) then
        -- Full info available
        hp = UnitHealth(healingTargetUnitID) / UnitHealthMax(healingTargetUnitID);
        hppre = (UnitHealth(healingTargetUnitID)+incommingHeal) / UnitHealthMax(healingTargetUnitID);
        hppost = (UnitHealth(healingTargetUnitID)+incommingHeal+self.Monitor.Size) / UnitHealthMax(healingTargetUnitID);
    else
        -- Estimate
        hp = UnitHealth(healingTargetUnitID)/100;
        local factor = hp / self:EstimateUnitHealth(healingTargetUnitID);
        hppre = hp + incommingHeal * factor;
        hppost = hp + (incommingHeal + self.Monitor.Size) * factor;
    end

    -- Determine waste (in percent)
    if (hppost > 1 and hppost > hppre) then
        waste = (hppost - 1) / (hppost - hppre);
    else
        waste = 0;
    end
    if (waste > 1) then 
        waste = 1;
    end

    -- Calculate colour for overheal severity
    local red = waste > 0.1 and 1 or waste * 10;
    local green = waste < 0.1 and 1 or -2.5 * waste + 1.25;
    if (waste < 0) then 
        green = 1;
        red = 0;
    end

    -- Sanity check on values
    if (hp > 1) then 
        hp = 1; 
    end
    if (hppre > 2) then 
        hpre = 2;
    end
    if (hppost > 3) then 
        hppost = 3;
    end
    if (hp > hppre) then 
        hppre = hp;
    end
    if (hppre > hppost) then 
        hppost = hppre;
    end    

    -- Update bars
    VisualHealBarStatusBar:SetValue(hp);
    VisualHealBarStatusBarPre:SetValue(hppre);
    VisualHealBarStatusBarPost:SetValue(hppost);
    VisualHealBarSparkOne:SetPoint("CENTER", "VisualHealBarStatusBar", "LEFT", 372/2 * hp, 0)
    VisualHealBarSparkTwo:SetPoint("CENTER", "VisualHealBarStatusBar", "LEFT", 372/2 * hppre, 0)

    -- Set colour for health
    VisualHealBarStatusBar:SetStatusBarColor(hp < 0.5 and 1 or 2 * (1 - hp), hp > 0.5 and 0.8 or 1.6 * hp, 0);

    -- Set colour for heal
    VisualHealBarStatusBarPost:SetStatusBarColor(red, green, 0)
end

--[ Monitor Functions ]--

function VisualHeal:StartMonitor()
    if (VisualHealOptions.ShowHealBar) then
        self.Monitor.IsRunning = true;
        this:RegisterEvent("UNIT_HEALTH");
        local _, class = UnitClass(self.Monitor.HealingTargetUnitID);
        VisualHealBarText:SetText((self.ClassColours[class] or "") .. self.Monitor.HealingTargetName);
        self:UpdateHealBar();
        VisualHealBar:Show();
    end
end

function VisualHeal:StopMonitor()
    this:UnregisterEvent("UNIT_HEALTH");
    VisualHealBar:Hide();
    self.Monitor.IsRunning = false;
    self:ResetMonitor();
end

function VisualHeal:ResetMonitor()
    self.Monitor.Spell = nil;
    self.Monitor.Rank = nil;
    self.Monitor.HealingTargetName = nil;
    self.Monitor.HealingTargetUnitID = nil;
    self.Monitor.Time = nil;
    self.Monitor.Size = nil;
end

--[ Event Handlers ]--

function VisualHeal:VARIABLES_LOADED()

    self.ClassModule = self:GetUnitClassModule('player');

    -- Unload everything if not a healer
    if (not self.ClassModule) then
        VisualHeal = nil;
        return;
    end

    -- Unload unused class modules
    VisualHeal.Druid = nil;
    VisualHeal.Paladin = nil;
    VisualHeal.Priest = nil;
    VisualHeal.Shaman = nil;

    -- Setup VisualHealVariables (and initialise upon first use)
    for k in pairs(DefaultOptions) do
        if (VisualHealOptions[k] == nil) then 
            VisualHealOptions[k] = DefaultOptions[k];
        end
    end

    -- Listen to spell events
    this:RegisterEvent("LEARNED_SPELL_IN_TAB");
    this:RegisterEvent("UNIT_SPELLCAST_SENT");
    this:RegisterEvent("UNIT_SPELLCAST_START");
    this:RegisterEvent("UNIT_SPELLCAST_STOP");
    this:RegisterEvent("UNIT_SPELLCAST_DELAYED");
    this:RegisterEvent("CHAT_MSG_ADDON");

    self:Print(self.Data.Name .. " " .. self.Data.Version .. " for " .. UnitClass('player') .. " Loaded");
end

function VisualHeal:GetUnitClassModule(unit)
    local _,unitClass = UnitClass(unit);
    unitClass = string.lower(unitClass);

    if (unitClass == "druid") then
        return VisualHeal.Druid;
    elseif (unitClass == "paladin") then
        return VisualHeal.Paladin;
    elseif (unitClass == "priest") then
        return VisualHeal.Priest;
    elseif (unitClass == "shaman") then
        return VisualHeal.Shaman;
    end
end

function VisualHeal:LEARNED_SPELL_IN_TAB()
    -- Invalidate cached data when learning new spells
    self.SpellCache = {};    
end

function VisualHeal:UNIT_SPELLCAST_SENT()
    if (UnitIsUnit('player', arg1)) then
        if (self.ClassModule.HealingSpells[arg2]) then
            -- Player is casting a healing spell
            self.Monitor.HealingTargetName = arg4;               
            self.Monitor.HealingTargetUnitID = self:UnitNameToUnitID(arg4);
            if (self.Monitor.HealingTargetUnitID) then
                self.Monitor.Spell = arg2;
                self.Monitor.Rank = tonumber(arg3:match("(%d+)"));
                self.Monitor.Time = nil;
                self.Monitor.Size = self.ClassModule:GetSpellInfo(self.Monitor.Spell, self.Monitor.Rank, self.Monitor.HealingTargetUnitID).EffectiveHeal;
                if (not self.Monitor.Size) then
                    -- Could not determine heal size
                    self:ResetMonitor();
                    return;
                end
                if (self.Debug) then
                    self:MultiPrint("SENT", self.Monitor.HealingTargetName, self.Monitor.HealingTargetUnitID, self.Monitor.Spell, self.Monitor.Rank, self.Monitor.Size);
                end
            else
                -- Could not identify healing target unit id
                self:ResetMonitor();
                return;
            end
        end
    end
end

function VisualHeal:UNIT_SPELLCAST_START()
    if (UnitIsUnit('player', arg1)) then
        if (self.Monitor.HealingTargetUnitID) then
            -- Continuation of healing spell cast
            local _, _, _, _, _, endTime = UnitCastingInfo(arg1);
            self.Monitor.Time = endTime / 1000;
            self:StartMonitor();
            self:CommSend();
            if (self.Debug) then
                self:MultiPrint("START", endTime / 1000);
            end
        end
    end
end

function VisualHeal:CHAT_MSG_ADDON()
    if (arg1 == "VisualHeal") then
        if (arg4 ~= UnitName('player')) then -- filter out own heal messages
            local _, _, commVersion, targetName, healSize, timeLeft = string.find(arg2, "^(%d+):(.+):(%d+):(%d+%.?%d*)");

            -- Ignore incompatible messages
            if (tonumber(commVersion) ~= self.Data.CommVersion) then
                return;
            end

            healSize = tonumber(healSize);
            timeLeft = tonumber(timeLeft);

            self:EntryUpdate(arg4, targetName, healSize, GetTime() + timeLeft);

            if (self.Monitor.IsRunning and (self.Monitor.HealingTargetName == targetName)) then
                self:UpdateHealBar();
            end
        end
    end
end

function VisualHeal:UNIT_HEALTH()
    if (UnitIsUnit(self.Monitor.HealingTargetUnitID, arg1)) then
        -- Update the heal bar when unit health changes on the healing target
        self:UpdateHealBar();
    end
end

function VisualHeal:UNIT_SPELLCAST_DELAYED()
    if (UnitIsUnit('player', arg1)) then
        if (self.Monitor.IsRunning) then
            -- The healing spell being cast has been delayed
            local _, _, _, _, _, endTime = UnitCastingInfo(arg1);
            self.Monitor.Time = endTime / 1000;
            if (self.Debug) then
                self:MultiPrint("DELAY", endTime / 1000);
            end
        end
    else
        -- The healing spell being cast by someone else has been delayed
        local _, _, _, _, _, endTime = UnitCastingInfo(arg1);
        if (endTime) then
            self:EntryUpdate(UnitName(arg1), nil, nil, endTime / 1000);
        end
    end

    -- Update the heal bar if monitor is running
    if (self.Monitor.IsRunning) then
        self:UpdateHealBar();
    end
end

function VisualHeal:UNIT_SPELLCAST_STOP()
    if (UnitIsUnit('player', arg1)) then
        self:StopMonitor();
        if (self.Debug) then
            self:MultiPrint("STOP");
        end
    else
        -- Delete the entry (if it exists)
        self:EntryDelete(UnitName(arg1));
    end
end
