--[[
	KeyBound
		An intuitive keybindings sytem
		Based off of ClickBinder by Gello and TrinityBinder by Maul/Osk
--]]

--local key, button
local enabled
local KeyBound
local currentBindings = {}
local msg = function(message) ChatFrame1:AddMessage(message or 'nil', 0,1,0.4) end

--[[ Conversions ]]--

local function Binder_ToBinding(button)
	return 'CLICK ' .. button:GetName() .. ':LeftButton'
end

local function Binder_GetBinding(button)
	return GetBindingKey(Binder_ToBinding(button))
end

--[[ Binding Functions ]]--

local function Binder_Unbind(button)
	if not InCombatLockdown() then
		if button then
			local binding = Binder_ToBinding(button)
			while GetBindingKey(binding) do
				SetBinding(GetBindingKey(binding), nil)
			end
			SaveBindings(GetCurrentBindingSet())
		end
	else
		msg(BONGOS_CANNOT_BIND_IN_COMBAT)
	end
end

local function Binder_SetBinding(button, key)
	if not InCombatLockdown() then
		if button and key then
			SetBindingClick(key, button:GetName(), 'LeftButton')
			SaveBindings(GetCurrentBindingSet())
			BMsg(format(BONGOS_KEY_BOUND_TO_ACTION, GetBindingText(key, 'KEY_'), button:GetName()))
		end
	else
		msg(BONGOS_CANNOT_BIND_IN_COMBAT)
	end
end

local function Binder_GetAllBindings(button)
	--[[ To figure out all the hotkeys for a spell, you need to treat them as a stack ]]--
	local found
	local binding = Binder_ToBinding(button)

	while GetBindingKey(binding) do
		found = true

		local hotKey = GetBindingKey(binding)
		table.insert(currentBindings, hotKey)
		SetBinding(hotKey, nil)
	end

	if found then
		local keys
		for i, hotKey in ipairs(currentBindings) do
			currentBindings[i] = nil
			SetBindingClick(hotKey, button:GetName(), 'LeftButton')

			if keys then
				keys = keys .. ', ' .. GetBindingText(hotKey,'KEY_')
			else
				keys = GetBindingText(hotKey,'KEY_')
			end
		end
		return keys
	end

	return BONGOS_BINDINGS_NO_KEYS_BOUND
end

--[[ OnX Functions ]]--

function Binder_OnKeyDown(binder, newKey)
	local button = binder.button
	
	if newKey == 'UNKNOWN' or newKey == 'SHIFT' or newKey == 'CTRL' or newKey == 'ALT' then return end
	
	local screenshotKey = GetBindingKey('SCREENSHOT')
	if screenshotKey and newKey == screenshotKey then
		Screenshot()
		return
	end
	
	local openChatKey = GetBindingKey('OPENCHAT')
	if openChatKey and newKey == openChatKey then
		ChatFrameEditBox:Show()
		return
	end

	if newKey == "LeftButton" or newKey == "RightButton" then
		return
	elseif newKey == "MiddleButton" then
		newKey = "BUTTON3"
	elseif newKey == "Button4" then
		newKey = "BUTTON4"
	elseif newKey == "Button5" then
		newKey = "BUTTON5"
	end
	
	if newKey == 'ESCAPE' then
		if button then
			Binder_Unbind(button)
			KeyBound_SetButton(button)
			msg(format(BONGOS_BINDINGS_CLEARED, button:GetName()))
		end
		return
	end

	local key = newKey
	if IsShiftKeyDown() then
		key = 'SHIFT-' .. key
	end
	if IsControlKeyDown() then
		key = 'CTRL-' .. key
	end
	if IsAltKeyDown() then
		key = 'ALT-' .. key
	end
	
	if button then
		local prevAction = GetBindingAction(key)
		if prevAction and prevAction ~= '' and prevAction ~= Binder_ToBinding(button) then
			msg(format(BONGOS_BINDINGS_UNBOUND_FROM_ACTION, GetBindingText(key, 'KEY_'), prevAction))
		end
		Binder_SetBinding(button, key)
		KeyBound_SetButton(button)
	end
end

local function Binder_OnEnter(self)
	local button = self.button
	if button and not InCombatLockdown() then
		if self:GetRight() >= (GetScreenWidth() / 2) then
			GameTooltip:SetOwner(self, "ANCHOR_LEFT")
		else
			GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
		end

		GameTooltip:SetText(button:GetName(), 1, 1, 1)
		GameTooltip:AddLine(Binder_GetAllBindings(button), 0, 1, 0)
		GameTooltip:AddLine(BONGOS_BINDINGS_CLEAR_HELP)
		GameTooltip:Show()
	else
		GameTooltip:Hide()
	end
end

local function Binder_Create()
	local binder = CreateFrame('Button')
	binder:RegisterForClicks('anyUp')
	binder:SetFrameStrata('DIALOG')
	binder:EnableKeyboard(true)
	binder:EnableMouseWheel(true)

	local bg = binder:CreateTexture()
	bg:SetTexture(0, 0, 0, 0.5)
	bg:SetAllPoints(binder)

	local text = binder:CreateFontString('OVERLAY')
	text:SetFontObject('GameFontNormalLarge')
	text:SetTextColor(0, 1, 0)
	text:SetAllPoints(binder)
	binder.text = text

	binder:SetScript('OnClick', function() Binder_OnKeyDown(this, arg1) end)
	binder:SetScript('OnKeyDown', function() Binder_OnKeyDown(this, arg1) end)
	binder:SetScript('OnMouseWheel', function()
		if arg1 > 0 then
			Binder_OnKeyDown(this, 'MOUSEWHEELUP')
		else
			Binder_OnKeyDown(this, 'MOUSEWHEELDOWN')
		end
	end)
	binder:SetScript('OnEnter', function() Binder_OnEnter(this) end)
	binder:SetScript('OnLeave', function() KeyBound_SetButton() GameTooltip:Hide() end)
	binder:SetScript('OnHide', function() KeyBound_SetButton() end)
	binder:Hide()
	
	return binder
end

--[[ Usable Functions ]]--

function KeyBound_SetButton(button)
	if enabled and button then
		KeyBound.button = button
		KeyBound.text:SetFontObject('GameFontNormalLarge')
		KeyBound.text:SetText(BBasicActionButton.GetHotkey(button))
		KeyBound:SetAllPoints(button)
		if KeyBound.text:GetStringWidth() > KeyBound:GetWidth() then
			KeyBound.text:SetFontObject('GameFontNormal')
		end
		KeyBound:Show()

		Binder_OnEnter(KeyBound)
	elseif KeyBound then
		KeyBound.button = nil
		KeyBound:ClearAllPoints()
		KeyBound:Hide()
	end
end

function KeyBound_Enable()
	enabled = true
	if not KeyBound then
		KeyBound = Binder_Create()
	end
	KeyBound_SetButton()
end

function KeyBound_Disable()
	enabled = nil
	KeyBound_SetButton()
end

--[[ Events ]]--

local wasEnabled
BEvent:AddAction('PLAYER_REGEN_ENABLED', function()
	if wasEnabled or enabled then
		KeyBound_Enable()
		msg(BONGOS_BINDINGS_COMBAT_ENABLED)
	end
	wasEnabled = nil
end)

BEvent:AddAction('PLAYER_REGEN_DISABLED', function()
	wasEnabled = enabled
	if wasEnabled then
		KeyBound_Disable()
		msg(BONGOS_BINDINGS_COMBAT_DISABLED)
	end
end)


--[[ Make things bindable ]]--

--spellbook
hooksecurefunc('SpellButton_OnEnter', function()
	if not IsPassiveSpell(this:GetID(), SpellBookFrame.bookType) then
		KeyBound_SetButton(this)
	end
end)

--action buttons
local oBActionButton_OnEnter = BActionButton.OnEnter
BActionButton.OnEnter = function(self, ...)
	oBActionButton_OnEnter(self, ...)
	KeyBound_SetButton(self)
end

--class buttons
local oBClassButton_OnEnter = BClassButton.OnEnter
BClassButton.OnEnter = function(self, ...)
	oBClassButton_OnEnter(self, ...)
	KeyBound_SetButton(self)
end

--pet buttons
local oBPetButton_OnEnter = BPetButton.OnEnter
BPetButton.OnEnter = function(self, ...)
	oBPetButton_OnEnter(self, ...)
	KeyBound_SetButton(self)
end