--[[
	BActionButton
		Scripts for Bongos Action Buttons, which add a few more features to blizzard action buttons

	Changes over the normal action buttons:
		Colored red when out of range
		Hotkeys are shortened
--]]

BActionButton = CreateFrame('CheckButton')
local Button_mt = {__index = BActionButton}

local SIZE = 36
local MAX_BUTTONS = 120 --the current maximum amount of action buttons
local buttons = {}

--[[ Local Button Events ]]--

local function OnUpdate() this:OnUpdate(arg1) end
local function PostClick() this:PostClick() end
local function OnDragStart() this:OnDragStart() end
local function OnReceiveDrag() this:OnReceiveDrag() end
local function OnEnter() this:OnEnter() end
local function OnLeave() this:OnLeave() end
local function OnShow() this:UpdateAll() end

local function OnAttributeChanged()
	if arg1 ~= 'statehidden' then
		local parent = this:GetParent()
		if parent then	
			local bar = parent:GetParent()
			if (bar:CanPage() or bar:HasStance()) and bar:IsVisible() then
				this.stateChanged = true
				this:UpdateAll()
				return
			end
		end
	end
	this:HideGrid()
end

--[[ Action Button Methods ]]--

BActionButton.HasNormalTexture = true

--Create an Action Button with the given ID and parent
function BActionButton.Create(id)
	if id > MAX_BUTTONS then return end

	local name = 'BActionButton' .. id
	local button = CreateFrame('CheckButton', name, parent, 'SecureActionButtonTemplate, ActionButtonTemplate')
	setmetatable(button, Button_mt)

	button:RegisterForDrag("LeftButton", "RightButton")
	button:RegisterForClicks('anyUp')
	button:SetAttribute('type', 'action')
	button:SetAttribute('action', id)
	button:SetAttribute('useparent-statebutton', true)
	button:SetAttribute('useparent-unit', true)
	button:SetAttribute('checkselfcast', true)

	button:SetScript('OnAttributeChanged', OnAttributeChanged)
	button:SetScript('OnUpdate', OnUpdate)
	button:SetScript('PostClick', PostClick)
	button:SetScript('OnDragStart', OnDragStart)
	button:SetScript('OnReceiveDrag', OnReceiveDrag)
	button:SetScript('OnEnter', OnEnter)
	button:SetScript('OnLeave', OnLeave)
	button:SetScript('OnShow', OnShow)
	button:Style()
	
	if not BActionSets_HotkeysShown() then
		getglobal(name .. "HotKey"):Hide()
	end	
	if not BActionSets_MacrosShown() then
		getglobal(name .. "Name"):Hide()
	end
	
	buttons[id] = button

	return button
end

function BActionButton.Set(id, parent)
	local button = buttons[id] or BActionButton.Create(id)
	if button then
		local prevParent = button:GetParent()
		parent.driver:SetAttribute('addchild', button)
		
		button:SetAlpha(parent:GetAlpha())
		button:SetParent(parent.driver)
		button:SetFrameLevel(0)

		if prevParent ~= parent.driver and parent:IsShown() then
			button:UpdateAllStances()
			button:UpdateAllPages()
			button:UpdateAll()
		end
		return button
	end
end
	
function BActionButton:Release()
	self:SetParent(nil)
	self:ClearAllPoints()
	CQ.Hide(self)
end
	
function BActionButton:Style()
	local name = self:GetName()
	getglobal(name .. 'Icon'):SetTexCoord(0.06, 0.94, 0.06, 0.94)
	getglobal(name .. 'Border'):SetVertexColor(0, 1, 0, 0.6)
	getglobal(name .. 'NormalTexture'):SetVertexColor(1, 1, 1, 0.5)
end
	
function BActionButton:UpdateStance(stance)
	local parent = self:GetParent()
	if parent then
		local bar = parent:GetParent()
		local baseID = tonumber(self:GetAttribute('action') or 1)

		local stanceID = mod(baseID + bar:GetStanceOffset(stance), MAX_BUTTONS)
		if stanceID == 0 then stanceID = MAX_BUTTONS end

		if id == stanceID then
			self:SetAttribute('*action-s' .. stance, nil)
		else
			self:SetAttribute('*action-s' .. stance, stanceID)
		end
	end
end

function BActionButton:UpdateAllStances()
	local parent = self:GetParent()
	if parent then
		local bar = parent:GetParent()
		local stances = bar:GetStances()

		if stances then
			for stance in pairs(stances) do
				self:UpdateStance(stance)
			end
		end
	end
end

function BActionButton:UpdatePage(page)
	local parent = self:GetParent()
	if parent then
		local bar = parent:GetParent()
		local baseID = tonumber(self:GetAttribute('action') or 1)

		local pagedID = mod(baseID + bar:GetPageOffset(page), MAX_BUTTONS)
		if pagedID == 0 then pagedID = MAX_BUTTONS end

		if baseID == pagedID then
			self:SetAttribute('*action-p' .. page, nil)
		else
			self:SetAttribute('*action-p' .. page, pagedID)
		end
	end
end

function BActionButton:UpdateAllPages()
	for i = 1, BActionBar.GetNumber() do
		self:UpdatePage(i)
	end
end

--[[ OnX Functions ]]--

function BActionButton:PostClick()
	if bg_showGrid then
		PickupAction(self:GetPagedID())
	end
	self:UpdateState()
end

function BActionButton:OnDragStart()
	if not BActionSets_ButtonsLocked() or (BActionSets_IsQuickMoveKeyDown() or bg_showGrid) then
		PickupAction(self:GetPagedID())
		self:UpdateState()
	end
end

function BActionButton:OnReceiveDrag()
	PlaceAction(self:GetPagedID())
	self:UpdateState()
end

function BActionButton:OnEnter()
	self:UpdateTooltip()
end

function BActionButton:OnLeave()
	self.updateTooltip = nil
	GameTooltip:Hide()
end

--OnUpdate
function BActionButton:OnUpdate(elapsed)
	local name = self:GetName()

	if not getglobal(name .. "Icon"):IsShown() then return end

	--update flashing
	if self.flashing == 1 then
		self.flashtime = self.flashtime - elapsed
		if self.flashtime <= 0 then
			local overtime = -self.flashtime
			if overtime >= ATTACK_BUTTON_FLASH_TIME then
				overtime = 0
			end
			self.flashtime = ATTACK_BUTTON_FLASH_TIME - overtime

			local flashTexture = getglobal(name .. "Flash")
			if flashTexture:IsShown() then
				flashTexture:Hide()
			else
				flashTexture:Show()
			end
		end
	end

	-- Handle range indicator
	if self.rangeTimer then
		if self.rangeTimer < 0 then
			local pagedID = self:GetPagedID()
			local hotkey = getglobal(name .. "HotKey")

			if IsActionInRange(pagedID) == 0 then
				hotkey:SetVertexColor(1, 0.1, 0.1)

				if BActionSets_ColorOutOfRange() and IsUsableAction(pagedID) then
					local r, g, b = BActionSets_GetRangeColor()
					getglobal(name .. "Icon"):SetVertexColor(r, g, b)
				end
			else
				hotkey:SetVertexColor(0.6, 0.6, 0.6)

				if IsUsableAction(pagedID) then
					getglobal(name .. "Icon"):SetVertexColor(1, 1, 1)
				end
			end
			self.rangeTimer = TOOLTIP_UPDATE_TIME
		else
			self.rangeTimer = self.rangeTimer - elapsed
		end
	end

	--Tooltip stuff, probably for the cooldown timer
	if self.updateTooltip then
		self.updateTooltip = self.updateTooltip - elapsed
		if self.updateTooltip <= 0 then
			if GameTooltip:IsOwned(self) then
				self:UpdateTooltip(self)
			else
				self.updateTooltip = nil
			end
		end
	end
end

--[[ Update Functions ]]--

--Updates the icon, count, cooldown, usability color, if the button is flashing, if the button is equipped,  and macro text.
function BActionButton:Update()
	local name = self:GetName()
	local action = self:GetPagedID()

	local icon = getglobal(name .. 'Icon')
	local cooldown = getglobal(name .. 'Cooldown')
	local texture = GetActionTexture(action)

	if texture then
		icon:SetTexture(texture)
		icon:Show()
		self.rangeTimer = -1
		
		if BActionButton.HasNormalTexture then
			self:SetNormalTexture('Interface\\Buttons\\UI-Quickslot2')
		end
	else
		icon:Hide()
		cooldown:Hide()
		self.rangeTimer = nil
		
		if BActionButton.HasNormalTexture then
			self:SetNormalTexture('Interface\\Buttons\\UI-Quickslot')
		end
		getglobal(name .. 'HotKey'):SetVertexColor(0.6, 0.6, 0.6)
	end

	if HasAction(action) then
		CQ.Show(self)

		self:UpdateState()
		self:UpdateUsable()
		self:UpdateCooldown()
		self:UpdateFlash()
	else
		if not(bg_showGrid or BActionSets_ShowGrid()) then
			CQ.Hide(self)
		else
			cooldown:Hide()
		end
	end	
	self:UpdateCount()

	-- Add a green border if button is an equipped item
	local border = getglobal(name .. 'Border')
	if IsEquippedAction(action) then
		border:SetVertexColor(0, 1, 0, 0.6)
		border:Show()
	else
		border:Hide()
	end

	if GameTooltip:IsOwned(self) then
		self:UpdateTooltip()
	else
		self.updateTooltip = nil
	end

	-- Update Macro Text
	getglobal(name .. 'Name'):SetText(GetActionText(action))
end

BActionButton.UpdateHotkey = BBasicActionButton.UpdateHotkey

--Update the cooldown timer
function BActionButton:UpdateCooldown()
	local start, duration, enable = GetActionCooldown(self:GetPagedID())
	CooldownFrame_SetTimer(getglobal(self:GetName().."Cooldown"), start, duration, enable)
end

function BActionButton:UpdateCount()
	local text = getglobal(self:GetName() .. 'Count')
	local action = self:GetPagedID()

	if IsConsumableAction(action) then
		text:SetText(GetActionCount(action))
	else
		text:SetText('')
	end
end

function BActionButton:StartFlash()
	self.flashing = 1
	self.flashtime = 0
	self:UpdateState()
end

function BActionButton:StopFlash()
	self.flashing = 0
	getglobal(self:GetName() .. 'Flash'):Hide()
	
	self:UpdateState()
end

function BActionButton:UpdateFlash()
	local action = self:GetPagedID()
	if (IsAttackAction(action) and IsCurrentAction(action)) or IsAutoRepeatAction(action) then
		self:StartFlash()
	else
		self:StopFlash()
	end
end

function BActionButton:UpdateTooltip()
	if BActionSets_TooltipsShown() then
		if GetCVar("UberTooltips") == "1" then
			GameTooltip_SetDefaultAnchor(GameTooltip, self)
		else
			GameTooltip:SetOwner(self, "ANCHOR_RIGHT")
		end

		local action = self:GetPagedID()
		if GameTooltip:SetAction(action) then
			self.updateTooltip = TOOLTIP_UPDATE_TIME
		else
			self.updateTooltip = nil
		end
	end
end

--Update if a button is checked or not
function BActionButton:UpdateState()
	local action = self:GetPagedID()
	self:SetChecked(IsCurrentAction(action) or IsAutoRepeatAction(action))
end

	--colors the action button if out of mana, out of range, etc
function BActionButton:UpdateUsable()
	local action = self:GetPagedID()
	local icon = getglobal(self:GetName().."Icon")

	local isUsable, notEnoughMana = IsUsableAction(action)
	if isUsable then
		if BActionSets_ColorOutOfRange() and IsActionInRange(action) == 0 then
			local r, g, b = BActionSets_GetRangeColor()
			icon:SetVertexColor(r, g, b)
		else
			icon:SetVertexColor(1, 1, 1)
		end
	elseif notEnoughMana then
		--Make the icon blue if out of mana
		icon:SetVertexColor(0.5, 0.5, 1)
	else
		--Skill unusable
		icon:SetVertexColor(0.3, 0.3, 0.3)
	end
end

function BActionButton:UpdateAll()
	self:Update()
	self:UpdateHotkey()
end

--Show Empty Buttons
function BActionButton:ShowGrid()
	CQ.Show(self)
end

--Hide Empty Buttons
function BActionButton:HideGrid()
	if not(bg_showGrid or BActionSets_ShowGrid() or HasAction(self:GetPagedID())) then
		CQ.Hide(self)
	end
end

function BActionButton:ShowMacroText(enable)
	if enable then
		getglobal(self:GetName().."Name"):Show()
	else
		getglobal(self:GetName().."Name"):Hide()
	end
end

--[[ Action ID Functions ]]--

--Data is cached because getting the current state of a button is a bit slow
local id = {}
function BActionButton:GetPagedID()
	if self:GetParent() then
		if self.stateChanged or not id[self] then
			self.stateChanged = nil
			id[self] = SecureButton_GetModifiedAttribute(self, 'action', SecureStateChild_GetEffectiveButton(self)) or 1
		end
		return id[self]
	end
	return 1
end

--[[ Meta Functions ]]--

function BActionButton.ForAll(action, ...)
	for _, button in pairs(buttons) do
		if button:GetParent() then
			action(button, ...)
		end
	end
end

function BActionButton.ForAllShown(action, ...)
	for _, button in pairs(buttons) do
		if button:GetParent() and button:IsShown() then
			action(button, ...)
		end
	end
end

function BActionButton.ForID(id, action, ...)
	for _, button in pairs(buttons) do
		if button:GetParent() and button:GetPagedID() == id then
			action(button, ...)
		end
	end
end

--[[ Utility Functions ]]--

function BActionButton.Get(id)
	return buttons[id]
end

function BActionButton.GetMax()
	return MAX_BUTTONS
end

function BActionButton.GetSize()
	return SIZE
end