--[[
	BActionBar
		A Bongos Actionbar
--]]

BActionBar = CreateFrame('Button')
local Bar_mt = {__index = BActionBar}

local DEFAULT_SPACE = 2
local DEFAULT_COLS = 12
local DEFAULT_ACTIONBARS = 10
local MAX_BUTTONS = BActionButton.GetMax()
local BUTTON_SIZE = BActionButton.GetSize()

--[[  Rightclick Menu ]]--

local function CreateConfigMenu(name)
	local menu = CreateFrame("Button", name, UIParent, "BongosRightClickMenu")
	menu:SetWidth(220)
	menu:SetHeight(282)

	local hideButton = CreateFrame("CheckButton", name .. "Hide", menu, "BongosHideButtonTemplate")

	local sizeSlider = CreateFrame("Slider", name .. "Size", menu, "BongosSlider")
	sizeSlider:SetPoint("TOPLEFT", hideButton, "BOTTOMLEFT", 2, -10)
	sizeSlider:SetScript("OnValueChanged", function()
		if not this:GetParent().onShow then
			local bar = this:GetParent().frame
			bar:Layout(nil, nil, this:GetValue())

			local size = bar:GetSize()
			BongosActionBarMenuColsLow:SetText(size)
			BongosActionBarMenuCols:SetMinMaxValues(1, size)
		end
		getglobal(this:GetName() .. "ValText"):SetText(this:GetValue())
	end)
	sizeSlider:SetValueStep(1)
	getglobal(name .. "SizeText"):SetText(BONGOS_SIZE)
	getglobal(name .. "SizeLow"):SetText(1)

	local colsSlider = CreateFrame("Slider", name .. "Cols", menu, "BongosSlider")
	colsSlider:SetPoint("TOP", sizeSlider, "BOTTOM", 0, -24)
	colsSlider:SetScript("OnValueChanged", function()
		local bar = this:GetParent().frame
		if not this:GetParent().onShow then
			bar:Layout(bar:GetSize() - this:GetValue() + 1)
		end
		getglobal(this:GetName() .. "ValText"):SetText(bar:GetSize() - this:GetValue() + 1)
	end)
	colsSlider:SetValueStep(1)
	getglobal(name .. "ColsText"):SetText(BONGOS_COLUMNS)
	getglobal(name .. "ColsHigh"):SetText(1)

	local spacingSlider = CreateFrame("Slider", name .. "Spacing", menu, "BongosSpaceSlider")
	spacingSlider:SetPoint("TOP", colsSlider, "BOTTOM", 0, -24)
	spacingSlider:SetScript("OnValueChanged", function()
		if not this:GetParent().onShow then
			local bar = this:GetParent().frame
			bar:Layout(nil, this:GetValue(), nil)
		end
		getglobal(this:GetName() .. "ValText"):SetText(this:GetValue())
	end)

	local scaleSlider = CreateFrame("Slider", name .. "Scale", menu, "BongosScaleSlider")
	scaleSlider:SetPoint("TOP", spacingSlider, "BOTTOM", 0, -24)

	local opacitySlider = CreateFrame("Slider", name .. "Opacity", menu, "BongosOpacitySlider")
	opacitySlider:SetPoint("TOP", scaleSlider, "BOTTOM", 0, -24)
end

--[[ Settings Loading ]]--

local function LoadMainBarDefaults()
	BActionSets[1] = {vis = 1, paging = 1}

	local class = select(2,UnitClass("player"))
	if class == 'DRUID' then
		BActionSets[1].stances = {8, nil, 6, nil, 9, nil, 7} --bear, aquatic, cat, travel, moonkin, flight, prowl
	elseif class == 'WARRIOR' then
		BActionSets[1].stances = {6, 7, 8}
	elseif class == 'ROGUE' then
		BActionSets[1].stances = {6}
	elseif class == 'PRIEST' then
		BActionSets[1].stances = {6}
	end
end

--[[ 
	ActionBar Methods 
--]]

function BActionBar.Create(id)
	if InCombatLockdown() then CQ.Do(function() BActionBar.Create(id) end) return end		
	if id == 1 and not BActionSets[1] then LoadMainBarDefaults() end

	local bar = BBar.Create(id, "BActionBar" .. id,  "BActionSets." .. id, BActionBar.ShowMenu, nil, BActionBar.Delete)
	setmetatable(bar, Bar_mt)
	bar:SetID(id)

	if not bar.driver then
		bar.driver = BState_Create(bar)
	end

	if not bar:IsUserPlaced() then
		local col = mod(bar:GetStart(), 12)
		if col == 0 then col = 12 end

		local row = ceil(bar:GetStart() / 12) - 1
		bar:SetPoint("BOTTOMLEFT", UIParent, "BOTTOM", -38*(12-col)+222, 38*row)
	end		
	bar:Layout()

	return bar
end

function BActionBar:Delete()
	for i = self:GetStart(), self:GetEnd() do
		button = BActionButton.Get(i)
		if button then
			button:Release()
		else
			break
		end
	end
end
	
function BActionBar:Layout(cols, space, size)
	if InCombatLockdown() then CQ.Do(function() self:Layout(cols, space, size) end) return end

	--[[ load settings ]]--
	
	local maxSize = self:GetMaxSize()
	if size == maxSize then
		self.sets.size = nil
	elseif size then
		self.sets.size = size
	end
	size = self:GetSize()

	if cols == DEFAULT_COLS then
		self.sets.cols = nil
	elseif cols then
		self.sets.cols = cols
	end
	cols = min(self.sets.cols or DEFAULT_COLS, size)

	if space == DEFAULT_SPACE then
		self.sets.space = nil
	elseif space then
		self.sets.space = space
	end
	space = self.sets.space or DEFAULT_SPACE

	--[[ Layout the Buttons ]]--
	
	if size < 1 then
		self:SetWidth(0); self:SetHeight(0)
	else
		local buttonSize = BUTTON_SIZE + space
		local offset = space / 2
		
		self:SetWidth(buttonSize * cols - space)
		self:SetHeight(buttonSize * ceil(size / cols) - space)

		local startID = self:GetStart()
		local endID = self:GetEnd()
		for i = startID, endID do
			local button = BActionButton.Set(i, self)
			if button then
				local normID = i - startID + 1
				local row = mod(normID - 1, cols)
				local col = ceil(normID / cols) - 1
				button:SetPoint('TOPLEFT', self, 'TOPLEFT', buttonSize * row, -buttonSize * col)
			else
				break
			end
		end
	end
	
	local maxID = self:GetStart() + self:GetMaxSize() - 1
	for i = self:GetEnd() + 1, maxID do
		local button = BActionButton.Get(i)
		if button then
			button:Release()
		else
			break
		end
	end
end

function BActionBar:SetSize(size)
	if size then
		self.sets.size = size
	end
	size = self:GetSize()
end

--[[
	Paging Functions

	Precedence:
		1. paged manually  return (BActionSets[id].page and CURRENT_ACTIONBAR_PAGE ~= 1)
			special key held down, shift + number
		2. paged via shapeshift return (BActionSets[id].switchOnShift and BActionMain.shapeshifted)
			in bear form, in cat form, in fury stance
		3. contextual paging
			should be used for such things as, if targeting a friendly unit,
--]]

--normal paging
function BActionBar:SetPaging(enable)
	if enable then
		self.sets.paging = 1
	else
		self.sets.paging = nil
	end
	self:UpdatePages()
end

function BActionBar:UpdatePages()
	for i = self:GetStart(), self:GetEnd() do
		local button = BActionButton.Get(i)
		if button then
			button:UpdateAllPages()
		else
			break
		end
	end
end

function BActionBar:GetPageOffset(page)
	if self:CanPage() then
		return ((page * (BActionSets.g.skip or 0) + page) * BActionBar.GetMaxSize()) or 0
	end
	return 0
end

function BActionBar:CanPage()
	return self.sets.paging
end

--[[ State Offsets - Handles things like which bar to go to when changing stances ]]--

function BActionBar:SetStanceOffset(stance, offset)
	local stances = self:GetStances()
	if offset and offset ~= 0 then
		if not stances then
			self.sets.stances = {}
			stances = self.sets.stances
		end
		stances[stance] = offset
	elseif stances then
		stances[stance] = nil
	end
	self:UpdateStance(stance)
end

function BActionBar:UpdateStance(stance)
	for i = self:GetStart(), self:GetEnd() do
		local button = BActionButton.Get(i)
		if button then
			button:UpdateStance(stance)
		else
			break
		end
	end
end

function BActionBar:UpdateStances()
	local stances = self:GetStances()
	if stances then
		for i = self:GetStart(), self:GetEnd() do
			local button = BActionButton.Get(i)
			if button then
				button:UpdateAllStances()
			else
				break
			end
		end
	end
end

function BActionBar:GetStanceOffset(stance)
	local stances = self:GetStances()
	if stances then
		local offset = stances[stance]
		if offset then
			return offset * BActionBar.GetMaxSize() or 0
		end
	end
	return 0
end

function BActionBar:GetStances()
	return self.sets.stances
end

function BActionBar:HasStance()
	local stances = self:GetStances()
	return stances and next(stances)
end

--[[ Start/End/Size ]]--

function BActionBar:GetSize()
	local maxSize = BActionBar.GetMaxSize()
	local size

	if self.sets then
		size = self.sets.size
	end

	return math.min(size or maxSize, maxSize)
end

function BActionBar:GetStart()
	return (self:GetID() - 1) * self:GetMaxSize() + 1
end

function BActionBar:GetEnd()
	return min(self:GetStart() + self:GetSize() - 1, MAX_BUTTONS)
end

function BActionBar.GetMaxSize()
	return floor(MAX_BUTTONS / BActionBar.GetNumber())
end

--[[
	Sets how many actionbars to use.  This function deletes all actionbars, then recreates them.
	This is done to make it easier to layout bars after adjusting how many there are.
--]]

function BActionBar.SetNumber(numBars)
	local diff = (numBars or DEFAULT_ACTIONBARS) - BActionBar.GetNumber()
	if diff ~= 0 then
		for i = 1, BActionBar.GetNumber() do
			BBar.Delete(i)
		end

		BActionSets.g.numActionBars = numBars
		for i = 1, BActionBar.GetNumber() do
			BActionBar.Create(i)
		end
		
		for i = 1, BActionBar.GetNumber() do
			local bar = BActionBar.Get(i)
			bar:UpdateStances()
			bar:UpdatePages()
		end
	end
end

function BActionBar.GetNumber()
	return BActionSets.g.numActionBars or DEFAULT_ACTIONBARS
end

--[[ Menu Functions ]]--

--Show the menu, loading all values
function BActionBar:ShowMenu()
	if not BongosActionBarMenu then
		CreateConfigMenu("BongosActionBarMenu")
	end

	BongosActionBarMenu.onShow = 1

	BongosActionBarMenu.frame = self

	BongosActionBarMenuText:SetText("Action Bar " .. self:GetID())
	BongosActionBarMenuHide:SetChecked(not self.sets.vis)

	local maxSize = self:GetMaxSize()
	local currentSize = self:GetSize()

	BongosActionBarMenuSizeHigh:SetText(maxSize)
	BongosActionBarMenuSize:SetMinMaxValues(1, maxSize)
	BongosActionBarMenuSize:SetValue(currentSize)

	BongosActionBarMenuCols:SetMinMaxValues(1, currentSize)
	BongosActionBarMenuColsLow:SetText(currentSize)
	BongosActionBarMenuCols:SetValue(currentSize - (self.sets.cols or DEFAULT_COLS) + 1)

	BongosActionBarMenuSpacing:SetValue(self.sets.space or DEFAULT_SPACE)
	BongosActionBarMenuScale:SetValue(self:GetScale() * 100)
	BongosActionBarMenuOpacity:SetValue(self:GetAlpha() * 100)

	BMenu.ShowForBar(BongosActionBarMenu, self)

	BongosActionBarMenu.onShow = nil
end
	
--[[ Utility ]]--
	
function BActionBar.Get(id)
	return getglobal('BActionBar' .. id)
end