--[[
	Gatherer Addon for World of Warcraft(tm).
	Version: 2.99.0.0394 (eagle)
	Revision: $Id: GatherMiniNotes.lua 392 2007-01-07 10:23:05Z norganna $

	License:
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program(see GPL.txt); if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is it's designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat

	Event handling routines
]]
local _tr = Gatherer.Locale.Tr
local _trC = Gatherer.Locale.TrClient
local _trL = Gatherer.Locale.TrLocale

-- reference to the Astrolabe mapping library
local Astrolabe = AceLibrary:GetInstance(Gatherer.AstrolabeVersion)

function Gatherer.MiniNotes.UpdateMinimapNotes(timeDelta, force)
	if (not Gatherer.Var.Loaded) then
		Gatherer.Util.Print("Gatherer not loaded")
		return
	end

	local setting = Gatherer.Command.GetSetting

	local needsUpdate = false
	if (not GatherNotes) then
		GatherNotes = { timeDiff=0, checkDiff=0 }
		needsUpdate = true
	else
		GatherNotes.timeDiff = GatherNotes.timeDiff + timeDelta
		if (GatherNotes.timeDiff > Gatherer.Var.NoteUpdateInterval * 10) then
			needsUpdate = true
		end
	end
	if (force) then
		needsUpdate = true
		recalculate = true
	end

	if (Gatherer.Command.IsUpdated("minimap.update")) then
		needsUpdate = true
	end

	if (needsUpdate) then
		local now = time()
		
		local c, z, px, py = Astrolabe:GetCurrentPlayerPosition()

		if not ( c and z and px and py ) then return end


		local curPos = 0
		local enabled = setting("minimap.enable")
		if (enabled == 1) then
			local maxDist = setting("minimap.distance", 800)
			local displayNumber = setting("minimap.count", 20)

			for i, nodeC,nodeZ, nodeID, nodePos, nodeDist, nodeX,nodeY, nodeCount, gtype, nodeHarvested, nodeInspected, nodeSource
			 in Gatherer.Storage.ClosestNodesInfo(c, z, px, py, displayNumber, maxDist, Gatherer.Command.DisplayFilter) do
				curPos = curPos + 1
				if (curPos > displayNumber) then break end

				-- need to position and label the corresponding button
				local gatherNote = getglobal("GatherNote"..curPos)
				local gatherNoteTexture = getglobal("GatherNote"..curPos.."Texture")

				local normSize = setting("minimap.iconsize")
				local normPerc = setting("minimap.percent") / 100
				local fadeEnab = setting("fade.enable") == 1
				local fadeDist = setting("fade.distance")
				local fadePerc = setting("fade.percent") / 100
				local tracEnab = setting("track.enable") == 1
				local tracCurr = setting("track.current") == 1
				local tracDist = setting("track.distance")
				local tracPerc = setting("track.percent") / 100
				local inspEnab = setting("inspect.enable") == 1
				local inspTint = setting("inspect.tint") == 1
				local inspDist = setting("inspect.distance")
				local inspPerc = setting("inspect.percent") / 100
				local inspTime = setting("inspect.time")
				local anonEnab = setting("anon.enable") == 1
				local anonTint = setting("anon.tint") == 1
				local anonPerc = setting("anon.percent") / 100
				
				local iconColor = "normal"
				local opacity = normPerc

				local selectedTexture, trimTexture = Gatherer.Util.GetNodeTexture(nodeID)

				-- If this icon has not been verified
				if (anonEnab and nodeSource and nodeSource ~= 'REQUIRE' and nodeSource ~= "IMPORTED") then
					opacity = anonPerc
					if anonTint then
						iconColor = "red"
					end
				end

				-- If node is within tracking distance
				if (tracEnab and nodeDist < tracDist) then
					if (not tracCurr or Gatherer.Util.GetTrackingMode() == Gatherer.Nodes.Objects[nodeID]) then
						selectedTexture = "Interface\\AddOns\\Gatherer\\Shaded\\White"
						opacity = tracPerc
						trimTexture = false
					end
				end

				-- If we need to fade the icon (because of great distance)
				if (fadeEnab) then
					if nodeDist >= fadeDist then
						opacity = fadePerc
					elseif nodeDist > tracDist then
						local range = math.max(fadeDist - tracDist, 0)
						local posit = math.min(nodeDist - tracDist, range)
						if (range > 0) then
							local ratio = posit / range
							local fadeLevel = (opacity - fadePerc) * ratio
							opacity = opacity - fadeLevel
						end
					end
				end

				-- If we are within inspect distance of this item, mark it as inspected
				if (inspEnab and nodeDist < inspDist) then
					Gatherer.Storage.SetNodeInspected(nodeC, nodeZ, nodeID, nodePos)
					if (inspTint) then
						iconColor = "green"
					end
					opacity = inspPerc

				-- If we've recently seen this node, set its transparency
				elseif (nodeInspected and inspTime > 0 and now - nodeInspected < inspTime) then
					local posit = math.max(now - nodeInspected, 0)
					local ratio = 1 - (posit / inspTime)
					local fadeLevel = (opacity - inspPerc) * ratio
					opacity = opacity - fadeLevel
				end

				-- Set the texture
				gatherNoteTexture:SetTexture(selectedTexture)
				gatherNote:SetWidth(normSize)
				gatherNote:SetHeight(normSize)

				-- Check to see if we need to trim the border off
				if (trimTexture) then
					gatherNoteTexture:SetTexCoord(0.08,0.92,0.08,0.92)
				else
					gatherNoteTexture:SetTexCoord(0,1,0,1)
				end

				-- Put the frame at the right level
				gatherNote:SetFrameLevel(MiniMapTrackingFrame:GetFrameLevel()-1)

				-- If this node is unverified, then make it reddish
				if (iconColor == "red") then
					gatherNoteTexture:SetVertexColor(0.9,0.4,0.4)
				elseif (iconColor == "green") then
					gatherNoteTexture:SetVertexColor(0.4,0.9,0.4)
				elseif (iconColor == "blue") then
					gatherNoteTexture:SetVertexColor(0.4,0.4,0.9)
				else
					gatherNoteTexture:SetVertexColor(1,1,1)
				end
				gatherNoteTexture:SetAlpha(opacity)

				if (not gatherNote.obj) then gatherNote.obj = {} end
				gatherNote.obj.id = nodeID
				gatherNote.obj.count = nodeCount
				gatherNote.obj.who = nodeSource
				gatherNote.obj.last = nodeInspected or nodeHarvested
				gatherNote.obj.dist = nodeDist

				Astrolabe:PlaceIconOnMinimap(gatherNote, nodeC, nodeZ, nodeX, nodeY)
			end
		end

		while (curPos < Gatherer.Var.MaxNumNotes) do
			curPos = curPos+1
			Astrolabe:RemoveIconFromMinimap(getglobal("GatherNote"..curPos));
		end
	end
end

-- Pass on any node clicks
function Gatherer.MiniNotes.MiniNoteOnClick()
	local x, y = GetCursorPosition()
	if ( Minimap.GetEffectiveScale ~= nil ) then
		x = x / Minimap:GetEffectiveScale()
		y = y / Minimap:GetEffectiveScale()
	else
		x = x / Minimap:GetScale()
		y = y / Minimap:GetScale()
	end

	local cx, cy = Minimap:GetCenter()
	x = x + CURSOR_OFFSET_X - cx
	y = y + CURSOR_OFFSET_Y - cy
	if ( sqrt(x * x + y * y) < (Minimap:GetWidth() / 2) ) then
		Minimap:PingLocation(x, y)
	end
end

function Gatherer.MiniNotes.MiniNoteOnEnter(frame)
	if (Gatherer.Settings and Gatherer.Settings.HideMiniNotes and Gatherer.Settings.HideMiniNotes == 1) then
		return
	end

	GameTooltip:SetOwner(this, "ANCHOR_BOTTOMLEFT")
	
	local id = frame.obj.id
	local name = Gatherer.Util.GetNodeName(id)
	local count = frame.obj.count
	local who = frame.obj.who
	local last = frame.obj.last
	local dist = frame.obj.dist

	GameTooltip:ClearLines()
	GameTooltip:AddLine(name)
	if (count > 0) then
		GameTooltip:AddLine(_tr("Gathered %1 times", count))
	end
	if (who) then
		if (who == "REQUIRE") then
			GameTooltip:AddLine(_tr("Unskilled Gather"))
		elseif (who == "IMPORTED") then
			GameTooltip:AddLine(_tr("Imported Gather"))
		else
			GameTooltip:AddLine(_tr("Source: %1", who))
		end
	end
	if (last and last > 0) then
		GameTooltip:AddLine(_tr("Last seen: %1s ago", time()-last))
	end
	GameTooltip:AddLine(_tr("Distance: %1 yards", dist))
	
	if (GatherDrops) then
		local drops = GatherDrops[id]
		if (drops) then
			local total = drops.total
			if (total and total > 0) then
				GameTooltip:AddLine(_tr("Overall drop rates:"))
				for item, count in pairs(drops) do
					if (item ~= "total") then
						local itemName, itemLink, itemRarity, itemLevel, itemMinLevel, itemType, itemSubType, itemStackCount, itemEquipLoc, invTexture = GetItemInfo(item) 
						
						if (itemName) then
							local expected = count/total
							if (expected > 0.01) then
								GameTooltip:AddDoubleLine(itemLink, string.format("x%0.2f", expected))
								GameTooltip:AddTexture(invTexture)
							end
						end
					end
				end
			end
		end
	end
	GameTooltip:Show()
end

