--[[
	Gatherer Addon for World of Warcraft(tm).
	Version: 2.99.0.0394 (eagle)
	Revision: $Id: GatherCommand.lua 392 2007-01-07 10:23:05Z norganna $

	License:
		This program is free software; you can redistribute it and/or
		modify it under the terms of the GNU General Public License
		as published by the Free Software Foundation; either version 2
		of the License, or (at your option) any later version.

		This program is distributed in the hope that it will be useful,
		but WITHOUT ANY WARRANTY; without even the implied warranty of
		MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
		GNU General Public License for more details.

		You should have received a copy of the GNU General Public License
		along with this program(see GPL.txt); if not, write to the Free Software
		Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is it's designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat

	Command parsing and processing
]]



local function getUserSig()
	local userSig = string.format("users.%s.%s", GetRealmName(), UnitName("player"))
	return userSig
end

local function getUserProfileName()
	local SETTINGS = Gatherer.Settings
	local userSig = getUserSig()
	return SETTINGS[userSig] or "Default"
end

local itc = 0
local function getUserProfile()
	local SETTINGS = Gatherer.Settings
	local profileName = getUserProfileName()
	if (not SETTINGS["profile."..profileName]) then
		if profileName ~= "Default" then
			profileName = "Default"
			SETTINGS[getUserSig()] = "Default"
		end
		if profileName == "Default" then
			SETTINGS["profile."..profileName] = {}
		end
	end
	return SETTINGS["profile."..profileName]
end

local function cleanse( source )
	for k in pairs(source) do
		source[k] = nil
	end
end

local updateTracker = {}
local function setUpdated()
	for k in pairs(updateTracker) do
		updateTracker[k] = nil
	end
end

function Gatherer.Command.IsUpdated(what)
	if not updateTracker[what] then
		updateTracker[what] = true
		return true
	end
	return false
end

local function getDefault(setting)
	local a,b,c = strsplit(".", setting)
	if (a == "show") then return 1 end
	if (b == "enable") then return 1 end
	if (setting == "mainmap.count")     then return 600   end
	if (setting == "mainmap.percent")   then return 80    end
	if (setting == "mainmap.iconsize")  then return 16    end
	if (setting == "minimap.count")     then return 20    end
	if (setting == "minimap.percent")   then return 80    end
	if (setting == "minimap.iconsize")  then return 16    end
	if (setting == "minimap.distance")  then return 800   end
	if (setting == "fade.distance")     then return 500   end
	if (setting == "fade.percent")      then return 20    end
	if (setting == "track.current")     then return 1     end
	if (setting == "track.distance")    then return 110   end
	if (setting == "track.percent")     then return 80    end
	if (setting == "inspect.tint")      then return 1     end
	if (setting == "inspect.distance")  then return 25    end
	if (setting == "inspect.percent")   then return 80    end
	if (setting == "inspect.time")      then return 120   end
	if (setting == "anon.tint")         then return true  end
	if (setting == "anon.percent")      then return 60    end
end

local function setter(setting, value)
	local SETTINGS = Gatherer.Settings
	local a,b,c = strsplit(".", setting)
	if (a == "profile") then
		local gui = Gatherer.Command.Gui
		if (setting == "profile.save") then
			value = gui.elements["profile.name"]:GetText()

			-- Create the new profile
			SETTINGS["profile."..value] = {}

			-- Set the current profile to the new profile
			SETTINGS[getUserSig()] = value
			-- Get the new current profile
			local newProfile = getUserProfile()
			-- Clean it out and then resave all data
			cleanse(newProfile)
			gui.Resave()

			-- Add the new profile to the profiles list
			local profiles = SETTINGS["profiles"]
			if (not profiles) then
				profiles = { "Default" }
				SETTINGS["profiles"] = profiles
			end
			-- Check to see if it already exists
			local found = false
			for pos, name in ipairs(profiles) do
				if (name == value) then found = true end
			end
			-- If not, add it and then sort it
			if (not found) then
				table.insert(profiles, value)
				table.sort(profiles)
			end
			DEFAULT_CHAT_FRAME:AddMessage("Saved profile: "..value)
		elseif (setting == "profile.delete") then
			-- User clicked the Delete button, see what the select box's value is.
			value = gui.elements["profile"].value

			-- If there's a profile name supplied
			if (value) then
				-- Clean it's profile container of values
				cleanse(SETTINGS["profile."..value])
				-- Delete it's profile container
				SETTINGS["profile."..value] = nil
				-- Find it's entry in the profiles list
				local profiles = SETTINGS["profiles"]
				if (profiles) then
					for pos, name in ipairs(profiles) do
						-- If this is it, then extract it
						if (name == value and name ~= "Default") then
							table.remove(profiles, pos)
						end
					end
				end
				-- If the user was using this one, then move them to Default
				if (getUserProfileName() == value) then
					SETTINGS[getUserSig()] = 'Default'
				end
				DEFAULT_CHAT_FRAME:AddMessage("Deleted profile: "..value)
			end
		elseif (setting == "profile") then
			-- User selected a different value in the select box, get it
			value = gui.elements["profile"].value

			-- Change the user's current profile to this new one
			SETTINGS[getUserSig()] = value
			DEFAULT_CHAT_FRAME:AddMessage("Changing profile: "..value)
		end

		-- Refresh all values to reflect current data
		gui.Refresh()
	else
		-- Set the value for this setting in the current profile
		local db = getUserProfile()
		db[setting] = value
		setUpdated()
	end
end

local function getter(setting)
	local SETTINGS = Gatherer.Settings
	if not setting then return end

	local a,b,c = strsplit(".", setting)
	if (a == 'profile') then
		if (b == 'profiles') then
			local pList = SETTINGS["profiles"]
			if (not pList) then
				pList = { "Default" }
			end
			return pList
		end
	end
	if (setting == 'profile') then
		return getUserProfileName()
	end
	local db = getUserProfile()
	return db[setting] or getDefault(setting)
end
function Gatherer.Command.GetSetting(setting, default)
	return getter(setting) or default
end

function Gatherer.Command.DisplayFilter(nodeId)
	local showType = "show."..Gatherer.Nodes.Objects[nodeId]:lower()
	local showObject = showType.."."..nodeId
	if ((getter(showType) or 1)==1 and (getter(showObject) or 1)==1) then
		return true
	end
	return false
end

function Gatherer.Command.MakeGuiConfig()
	if (Gatherer.Command.Gui) then return end

	local id, last, cont
	local gui = Configator.NewConfigator(setter, getter)
	Gatherer.Command.Gui = gui

	id = gui.AddTab("Profiles")
	gui.AddControl(id, "Header",     0,    "Setup, configure and edit profiles")
	gui.AddControl(id, "Subhead",    0,    "Activate a current profile")
	gui.AddControl(id, "Selectbox",  0, 1, "profile.profiles", "profile", "Switch to given profile")
	gui.AddControl(id, "Button",     0, 1, "profile.delete", "Delete")
	gui.AddControl(id, "Subhead",    0,    "Create or replace a profile")
	gui.AddControl(id, "Text",       0, 1, "profile.name", "New profile name:")
	gui.AddControl(id, "Button",     0, 1, "profile.save", "Save")
	
	id = gui.AddTab("Options")
	gui.AddControl(id, "Header",     0,    "Main Gatherer options")
	last = gui.GetLast(id) -- Get the current position so we can return here for the second column

	gui.AddControl(id, "Subhead",    0,    "Tracking options")
	gui.AddControl(id, "Checkbox",   0, 1, "show.mine", "Show mining nodes")
	gui.AddControl(id, "Checkbox",   0, 1, "show.herb", "Show herbalism nodes")
	gui.AddControl(id, "Checkbox",   0, 1, "show.open", "Show treasure nodes")

	gui.AddControl(id, "Subhead",    0,    "WorldMap options")
	gui.AddControl(id, "Checkbox",   0, 1, "mainmap.enable", "Display nodes on WorldMap")
	gui.AddControl(id, "Slider",     0, 2, "mainmap.count", 10, 1000, 10, "Display: %d nodes")
	gui.AddControl(id, "Slider",     0, 2, "mainmap.percent", 10, 100, 2, "Opacity: %d%%")
	gui.AddControl(id, "Slider",     0, 2, "mainmap.iconsize", 4, 64, 1, "Icon size: %d")

	gui.AddControl(id, "Subhead",    0,    "Minimap options")
	gui.AddControl(id, "Checkbox",   0, 1, "minimap.enable", "Display nodes on Minimap")
	gui.AddControl(id, "Slider",     0, 2, "minimap.count", 1, 50, 1, "Display: %d closest")
	gui.AddControl(id, "Slider",     0, 2, "minimap.percent", 0, 100, 1, "Default opacity: %d%%")
	gui.AddControl(id, "Slider",     0, 2, "minimap.iconsize", 4, 64, 1, "Icon size: %d")
	gui.AddControl(id, "Slider",     0, 2, "minimap.distance", 100, 5000, 50, "Distance: %d yards")

	gui.SetLast(id, last) -- Return to the saved position
	gui.AddControl(id, "Subhead",  0.5,    "Minimap additional")
	gui.AddControl(id, "Checkbox", 0.5, 1, "fade.enable", "Fade out mininotes");
	gui.AddControl(id, "Slider",   0.5, 2, "fade.distance", 10, 1000, 10, "Fade at: %d yards")
	gui.AddControl(id, "Slider",   0.5, 2, "fade.percent", 0, 100, 1, "Fade opacity: %d%%")
	gui.AddControl(id, "Checkbox", 0.5, 1, "track.enable", "Convert to circle when close");
	gui.AddControl(id, "Checkbox", 0.5, 2, "track.current", "Only for active tracking skill");
	gui.AddControl(id, "Slider",   0.5, 2, "track.distance", 50, 150, 1, "Convert at: %d yards")
	gui.AddControl(id, "Slider",   0.5, 2, "track.percent", 0, 100, 1, "Circle opacity: %d%%")
	gui.AddControl(id, "Checkbox", 0.5, 1, "inspect.enable", "Mark nodes as inspected");
	gui.AddControl(id, "Checkbox", 0.5, 2, "inspect.tint", "Tint green while inspecting");
	gui.AddControl(id, "Slider",   0.5, 2, "inspect.distance", 1, 100, 1, "Inspect at: %d yards")
	gui.AddControl(id, "Slider",   0.5, 2, "inspect.percent", 0, 100, 1, "Inspect opacity: %d%%")
	gui.AddControl(id, "Slider",   0.5, 2, "inspect.time", 10, 900, 10, "Reinspect: %d seconds")
	gui.AddControl(id, "Checkbox", 0.5, 1, "anon.enable", "Display anonymous nodes");
	gui.AddControl(id, "Checkbox", 0.5, 2, "anon.tint", "Tint anonymous nodes red");
	gui.AddControl(id, "Slider",   0.5, 2, "anon.percent", 0, 100, 1, "Anon opacity: %d%%");

	-- Get all objects and insert them into the appropriate subtable
	local itemLists = {}
	for name, objid in pairs(Gatherer.Nodes.Names) do
		local gtype = Gatherer.Nodes.Objects[objid]:lower()
		if (not itemLists[gtype]) then itemLists[gtype] = {} end
		local entry = { objid, name }
		local cat = Gatherer.Categories.ObjectCategories[objid]
		if (cat) then
			local skill = Gatherer.Constants.SkillLevel[cat]
			if (skill) then
				table.insert(entry, skill)
			end
		end
		table.insert(itemLists[gtype], entry)
	end

	function entrySort(a, b)
		if (b == nil) then return nil end

		local aName = a[2]
		local bName = b[2]
		local aLevel = a[3]
		local bLevel = b[3]

		if bLevel then
			if aLevel then
				if aLevel < bLevel then return true end
				if bLevel < aLevel then return false end
			else
				return true
			end
		elseif aLevel then
			return false
		end
		local comp = aName < bName
		return comp
	end

	-- Print out tabs and checkboxes for the above list
	id = gui.AddTab("Mining")
	gui.AddControl(id, "Header",     0,    "Mining filter options")
	gui.AddControl(id, "Subhead",    0,    "Mineral nodes to track")
	local options = {}
	local list = itemLists.mine
	table.sort(list, entrySort)
	for pos, mine in pairs(list) do
		table.insert(options, { "show.mine."..mine[1], mine[2] })
	end
	gui.ColumnCheckboxes(id, 2, options)


	id = gui.AddTab("Herbalism")
	gui.AddControl(id, "Header",     0,    "Herbalism filter options")
	gui.AddControl(id, "Subhead",    0,    "Herbal nodes to track")
	local options = {}
	local list = itemLists.herb
	table.sort(list, entrySort)
	for pos, herb in pairs(list) do
		table.insert(options, { "show.herb."..herb[1], herb[2] })
	end
	gui.ColumnCheckboxes(id, 3, options)

	id = gui.AddTab("Treasure")
	gui.AddControl(id, "Header",     0,    "Treasure filter options")
	last = gui.AddControl(id, "Subhead",    0,    "Treasure nodes to track")
	local options = {}
	local list = itemLists.open
	table.sort(list, entrySort)
	for pos, open in pairs(list) do
		table.insert(options, { "show.open."..open[1], open[2] })
	end
	gui.ColumnCheckboxes(id, 3, options)
end

function Gatherer.Command.GetFilterVal(gName)
	local filter = Gatherer.Command.GetFilter(gName)
	if (filter) then return "on" end
	return "off"
end

function Gatherer.Command.GetFilter(gName)
	local gType
	if (gName == "herbs") then gType = "herb" end
	if (gName == "mining") then gType = "mine" end
	if (gName == "treasures") then gType = "open" end
	if (not gType) then gType = gName end
	local db = getUserProfile()
	return db["show."..gType]
end


function Gatherer.Command.Process(command)
	Gatherer.Command.MakeGuiConfig()
	Gatherer.Command.Gui:Show()
	return
end


--[[
	if true then return end

	local SETTINGS = Gatherer.Settings
	local i,j, cmd, param = string.find(command, "^([^ ]+) (.+)$")
	if (not cmd) then cmd = command; end
	if (not cmd) then cmd = ""; end
	if (not param) then param = ""; end

	if ((cmd == "") or (cmd == "help")) then
		local useMinimap = "Off"
		if (SETTINGS.useMinimap) then useMinimap = "On"; end
		local useMainmap = "Off"
		if (SETTINGS.useMainmap) then useMainmap = "On"; end
		local mapMinder = "Off"
		if (SETTINGS.mapMinder) then mapMinder = "On"; end
		local minderTime = "5s"
		if (SETTINGS.minderTime) then minderTime = SETTINGS.minderTime.."s"; end

		Gatherer.Util.ChatPrint("Usage:")
		Gatherer.Util.ChatPrint("  |cffffffff/gather (on|off|toggle)|r |cff2040ff["..useMinimap.."]|r - turns the gather minimap display on and off")
		Gatherer.Util.ChatPrint("  |cffffffff/gather mainmap (on|off|toggle)|r |cff2040ff["..useMainmap.."]|r - turns the gather mainmap display on and off")
		Gatherer.Util.ChatPrint("  |cffffffff/gather minder (on|off|toggle|<n>)|r |cff2040ff["..mapMinder.."]|r - turns the gather map minder on and off (remembers and reopens your last open main map; within "..minderTime..")")
		Gatherer.Util.ChatPrint("  |cffffffff/gather dist <n>|r |cff2040ff["..SETTINGS.maxDist.."]|r - sets the maximum search distance for display (0=infinite(default), typical=10)")
		Gatherer.Util.ChatPrint("  |cffffffff/gather num <n>|r |cff2040ff["..SETTINGS.number.."]|r - sets the maximum number of items to display (default=10, up to 25)")
		Gatherer.Util.ChatPrint("  |cffffffff/gather fdist <n>|r |cff2040ff["..SETTINGS.fadeDist.."]|r - sets a fade distance (in units) for the icons to fade out by (default = 20)")
		Gatherer.Util.ChatPrint("  |cffffffff/gather fperc <n>|r |cff2040ff["..SETTINGS.fadePerc.."]|r - sets the percentage for fade at max fade distance (default = 80 [=80% faded])")
		Gatherer.Util.ChatPrint("  |cffffffff/gather theme <name>|r |cff2040ff["..SETTINGS.iconSet.."]|r - sets the icon theme: original, shaded (default), iconic or iconshade")
		Gatherer.Util.ChatPrint("  |cffffffff/gather idist <n>|r |cff2040ff["..SETTINGS.miniIconDist.."]|r - sets the minimap distance at which the gather icon will become iconic (0 = off, 1-60 = pixel radius on minimap, default = 40)")
		Gatherer.Util.ChatPrint("  |cffffffff/gather herbs (on|off|toggle|auto)|r |cff2040ff["..Gatherer.Command.GetFilterVal("herbs").."]|r - select whether to show herb data on the minimap")
		Gatherer.Util.ChatPrint("  |cffffffff/gather mining (on|off|toggle|auto)|r |cff2040ff["..Gatherer.Command.GetFilterVal("mining").."]|r - select whether to show mining data on the minimap")
		Gatherer.Util.ChatPrint("  |cffffffff/gather treasure (on|off|toggle|auto)|r |cff2040ff["..Gatherer.Command.GetFilterVal("treasure").."]|r - select whether to show treasure data on the minimap")
		Gatherer.Util.ChatPrint("  |cffffffff/gather options|r - show/hide UI Options dialog.")
		Gatherer.Util.ChatPrint("  |cffffffff/gather report|r - show/hide report dialog.")
		Gatherer.Util.ChatPrint("  |cffffffff/gather search|r - show/hide search dialog.")
		Gatherer.Util.ChatPrint("  |cffffffff/gather loginfo (on|off)|r - show/hide logon information.")
		Gatherer.Util.ChatPrint("  |cffffffff/gather filterrec (herbs|mining|treasure)|r - link display filter to recording for selected gathering type")
	elseif (cmd == "options" ) then
		if ( Gatherer_DialogFrame:IsVisible() ) then
			Gatherer.Interface.HideOptions()
		else
			Gatherer.Interface.ShowOptions()
		end
	elseif (cmd == "report" ) then
		showGathererInfo(1)
	elseif (cmd == "search" ) then
		showGathererInfo(2)
	elseif (cmd == "loginfo" ) then
		local value
		if (not param or param == "") then value = "on"; else value = param; end
		Gatherer.Util.ChatPrint("Setting log information display to "..value)
		SETTINGS.logInfo = value
	elseif ( cmd == "filterrec" ) then
		local value
		if (not param) then
			return
		end
		if ( param == "treasure" ) then
			value = "OPEN"
		elseif ( param == "herbs" ) then
			value = "HERB"
		elseif ( param == "mining" ) then
			value = "MINE"
		end

		if ( value > -1 ) then
			if ( SETTINGS.filterRecording[value] ) then
				SETTINGS.filterRecording[value] = nil
				Gatherer.Util.ChatPrint("Turned filter/recording link for "..param.." off.")
			else
				SETTINGS.filterRecording[value] = 1
				Gatherer.Util.ChatPrint("Turned filter/recording link for "..param.." on.")
			end
		end
	elseif (cmd == "on") then
		SETTINGS.useMinimap = true
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
		SETTINGS.useMinimapText = "on"
		Gatherer.Util.ChatPrint("Turned gather minimap display on")
	elseif (cmd == "off") then
		SETTINGS.useMinimap = false
		SETTINGS.useMinimapText = "off"
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
		Gatherer.Util.ChatPrint("Turned gather minimap display off (still collecting)")
	elseif (cmd == "toggle") then
		SETTINGS.useMinimap = not SETTINGS.useMinimap
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
		if (SETTINGS.useMinimap) then
			Gatherer.Util.ChatPrint("Turned gather minimap display on")
			SETTINGS.useMinimapText = "on"
		else
			Gatherer.Util.ChatPrint("Turned gather minimap display off (still collecting)")
			SETTINGS.useMinimapText = "off"
		end
	elseif (cmd == "dist") then
		local i,j, value = string.find(param, "(%d+)")
		if (not value) then value = 0; else value = value + 0.0; end
		if (value <= 0) then
			SETTINGS.maxDist = 0
		else
			SETTINGS.maxDist = value + 0.0
		end
		Gatherer.Util.ChatPrint("Setting maximum note distance to "..SETTINGS.maxDist)
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
	elseif (cmd == "fdist") then
		local i,j, value = string.find(param, "(%d+)")
		if (not value) then value = 0; else value = value + 0.0; end
		if (value <= 0) then
			SETTINGS.fadeDist = 0
		else
			SETTINGS.fadeDist = value + 0.0
		end
		Gatherer.Util.ChatPrint("Setting fade distance to "..SETTINGS.fadeDist)
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
	elseif (cmd == "fperc") then
		local i,j, value = string.find(param, "(%d+)")
		if (not value) then value = 0; else value = value + 0.0; end
		if (value <= 0) then
			SETTINGS.fadePerc = 0
		else
			SETTINGS.fadePerc = value + 0.0
		end
		Gatherer.Util.ChatPrint("Setting fade percent at fade distance to "..SETTINGS.fadePerc)
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
	elseif ((cmd == "idist") or (cmd == "icondist")) then
		local i,j, value = string.find(param, "(%d+)")
		if (not value) then value = 0; else value = value + 0; end
		if (value <= 0) then
			SETTINGS.miniIconDist = 0
		else
			SETTINGS.miniIconDist = value + 0
		end
		Gatherer.Util.ChatPrint("Setting iconic distance to "..SETTINGS.miniIconDist)
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
	elseif (cmd == "theme") then
		if (Gather_IconSet[param]) then
			SETTINGS.iconSet = param
			Gatherer.Util.ChatPrint("Gatherer theme set to "..SETTINGS.iconSet)
		else
			Gatherer.Util.ChatPrint("Unknown theme: "..param)
		end
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
	elseif ((cmd == "num") or (cmd == "number")) then
		local i,j, value = string.find(param, "(%d+)")
		if (not value) then value = 0; else value = value + 0; end
		if (value < 0) then
			SETTINGS.number = 10
		elseif (value <= Gatherer.Var.MaxNumNotes) then
			SETTINGS.number = math.floor(value + 0)
		else
			SETTINGS.number = Gatherer.Var.MaxNumNotes
		end
		if (SETTINGS.number == 0) then
			SETTINGS.useMinimap = false
			SETTINGS.useMinimapText = "off"
			Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
			Gatherer.Util.ChatPrint("Turned gather minimap display off (still collecting)")
		else
			if ((SETTINGS.number > 0) and (SETTINGS.useMinimap == false)) then
				SETTINGS.useMinimap = true
		        	SETTINGS.useMinimapText = "on"
				Gatherer.Util.ChatPrint("Turned gather minimap display on")
			end
			Gatherer.Util.ChatPrint("Displaying "..SETTINGS.number.." notes at once")
			Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
		end
	elseif (cmd == "mainmap") then
		if ((param == "false") or (param == "off") or (param == "no") or (param == "0")) then
			SETTINGS.useMainmap = false
		elseif (param == "toggle") then
			SETTINGS.useMainmap = not SETTINGS.useMainmap
		else
			SETTINGS.useMainmap = true
		end
		if (SETTINGS.useMainmap) then
			Gatherer.Util.ChatPrint("Displaying notes in main map")
			Gatherer_WorldMapDisplay:SetText("Hide Items")
		else
			Gatherer.Util.ChatPrint("Not displaying notes in main map")
			Gatherer_WorldMapDisplay:SetText("Show Items")
		end

		if (SETTINGS.useMainmap and SETTINGS.showWorldMapFilters and SETTINGS.showWorldMapFilters == 1) then
			GathererWD_DropDownFilters:Show()
		end

	elseif (cmd == "minder") then
		if ((param == "false") or (param == "off") or (param == "no") or (param == "0")) then
			SETTINGS.mapMinder = false
		elseif (param == "toggle") then
			SETTINGS.mapMinder = not SETTINGS.mapMinder
		elseif (param == "on") then
			SETTINGS.mapMinder = true
		else
			local i,j, value = string.find(param, "(%d+)")
			if (not value) then value = 0; else value = value + 0; end
			if (value <= 0) then
				SETTINGS.mapMinder = false
				SETTINGS.minderTime = 0
			else
				SETTINGS.mapMinder = true
				SETTINGS.minderTime = value + 0
			end
			Gatherer.Util.ChatPrint("Setting map minder timeout to "..SETTINGS.minderTime)
		end
		if (SETTINGS.mapMinder) then
			Gatherer.Util.ChatPrint("Map minder activated at "..SETTINGS.minderTime)
		else
			Gatherer.Util.ChatPrint("Not minding your map")
		end
	elseif ((cmd == "herbs") or (cmd == "mining") or (cmd == "treasure")) then
		if ((param == "false") or (param == "off") or (param == "no") or (param == "0")) then
			Gatherer.Command.SetFilter(cmd, "off")
			Gatherer.Util.ChatPrint("Not displaying "..cmd.." notes in minimap")
		elseif (param == "on" or param == "On" ) then
			Gatherer.Command.SetFilter(cmd, "on")
			Gatherer.Util.ChatPrint("Displaying "..cmd.." notes in minimap")
		elseif (param == "toggle" or param == "") then
			local cur = Gatherer.Command.GetFilterVal(cmd)
			if ((cur == "on") or (cur == "auto")) then
				Gatherer.Command.SetFilter(cmd, "off")
				Gatherer.Util.ChatPrint("Not displaying "..cmd.." notes in minimap")
			else
				Gatherer.Command.SetFilter(cmd, "on")
				Gatherer.Util.ChatPrint("Displaying "..cmd.." notes in minimap")
			end
		else
			Gatherer.Command.SetFilter(cmd, "auto")
			Gatherer.Util.ChatPrint("Displaying "..cmd.." notes in minimap based on ability")
		end
		Gatherer.MiniNotes.UpdateMinimapNotes(0, true)
		Gatherer.MapNotes.MapDraw()
	end
end
if true then return end

function Gatherer.Command.SetFilter(gatherType, value)
	if ( gatherType == "treasure" ) then
		gatherType = "OPEN"
	elseif ( gatherType == "herbs" ) then
		gatherType = "HERB"
	elseif ( gatherType == "mining" ) then
		gatherType = "MINE"
	end
	if ( value == "on" or value == "off" or value == "auto" ) then
		--setting is valid, do nothing
	elseif ( value ) then
		value = "on"
	else
		value = "auto"
	end
	Gatherer.Settings.filters[gatherType] = value
end

function Gatherer.Command.GetFilterVal(type)
	return Gatherer.Settings.filters[type] or "auto"
end

function Gatherer.Command.GetFilter(filter)
	local value = Gatherer.Command.GetFilterVal(filter)
	local filterVal = false

	if (value == "on") then
		filterVal = true
	elseif (value == "off") then
		filterVal = false
	elseif (value == "auto") then
		if (filter == "OPEN") then
			filterVal = true
		end
		if (not Gatherer.Var.Skills) then
			filterVal = true
		end
		if ((Gatherer.Var.Skills[filter]) and (Gatherer.Var.Skills[filter] > 0)) then
			filterVal = true
		end
	end

	return filterVal
end


]]
