--[[
	Gatherer Addon for World of Warcraft(tm).
	Version: 2.99.0.0331 (eagle)
	Revision: $Id: GatherEvent.lua 329 2006-12-04 03:30:31Z norganna $

	License:
	This program is free software; you can redistribute it and/or
	modify it under the terms of the GNU General Public License
	as published by the Free Software Foundation; either version 2
	of the License, or (at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program(see GPL.txt); if not, write to the Free Software
	Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

	Note:
		This AddOn's source code is specifically designed to work with
		World of Warcraft's interpreted AddOn system.
		You have an implicit licence to use this AddOn with these facilities
		since that is it's designated purpose as per:
		http://www.fsf.org/licensing/licenses/gpl-faq.html#InterpreterIncompat

	Event handling routines
]]
local _tr = Gatherer.Locale.Tr
local _trC = Gatherer.Locale.TrClient
local _trL = Gatherer.Locale.TrLocale

-- reference to the Astrolabe mapping library
local Astrolabe = AceLibrary:GetInstance(Gatherer.AstrolabeVersion)

function Gatherer.Event.OnLoad()
	this:RegisterEvent("WORLD_MAP_UPDATE")
	this:RegisterEvent("CLOSE_WORLD_MAP"); -- never triggered apparently
	this:RegisterEvent("LEARNED_SPELL_IN_TAB"); -- follow current skills
	this:RegisterEvent("SPELLS_CHANGED"); -- follow current skills
	this:RegisterEvent("SKILL_LINES_CHANGED"); -- follow current skills

	-- Events for off world non processing
	this:RegisterEvent("PLAYER_ENTERING_WORLD")
	this:RegisterEvent("PLAYER_LEAVING_WORLD")

	-- Addon Loaded and player login/logout events
	this:RegisterEvent("ADDON_LOADED")
	this:RegisterEvent("PLAYER_LOGIN")
	this:RegisterEvent("PLAYER_LOGOUT")

	-- Communications
	this:RegisterEvent("CHAT_MSG_ADDON")

	Gatherer.Mapping.LoadZoneData()

	SLASH_GATHER1 = "/gather"
	SLASH_GATHER2 = "/gatherer"
	SlashCmdList["GATHER"] = function(msg)
		Gatherer.Command.Process(msg)
	end

	LibSwag.RegisterHook("Gatherer", Gatherer.Event.OnSwag)
end

function Gatherer.Event.OnEvent(...)
	-- Enable/Disable event processing for zoning
	if (event == "PLAYER_ENTERING_WORLD" ) then
		GatherNote_UpdateFrame:Show()

	elseif (event == "PLAYER_LEAVING_WORLD" ) then
		GatherNote_UpdateFrame:Hide()

	elseif (event == "WORLD_MAP_UPDATE") then
		if (WorldMapFrame:IsVisible()) then
			local serverTime = GetTime()
			if (Gatherer.Settings.mapMinder == true and Gatherer.Var.MapOpen == false) then
				-- Enhancement to open to last opened map if we were there less than a minute ago
				-- Otherwise, go to the player's current position
				local startContinent, startZone
				if (Gatherer.Var.CloseMap and (serverTime - Gatherer.Var.CloseMap.time < Gatherer.Settings.minderTime)) then
					startContinent = Gatherer.Var.CloseMap.continent
					startZone = Gatherer.Var.CloseMap.zone
				else
					startContinent, startZone = Gatherer.Mapping.GetCurrentZone()
				end

				Gatherer.Var.MapOpen = true
				if ( GetCurrentMapContinent()>0 and startZone and startZone > 0 ) then
					SetMapZoom(startContinent, startZone)
				end
			end
			Gatherer.Var.MapOpen = true
			local mapContinent = GetCurrentMapContinent()
			local mapZone = GetCurrentMapZone()
			Gatherer.Var.CloseMap = { continent = mapContinent, zone = mapZone, time = GetTime() }
			Gatherer.Mapping.MapDraw()
		elseif (Gatherer.Var.MapOpen) then
			Gatherer.Var.MapOpen = false
			Gatherer.Mapping.ChangeMap()
			Gatherer.Mapping.MapDraw()
		end
	elseif ( event == "CLOSE_WORLD_MAP") then
		-- never called apparently
		Gatherer.Var.MapOpen = false

		Gatherer.Mapping.ChangeMap()
		Gatherer.Mapping.MapDraw()

	elseif( event == "ADDON_LOADED") then
		if ( myAddOnsFrame_Register ) then
			-- myAddons Support
			GathererDetails["name"] = "Gatherer"
			GathererDetails["version"] = GATHERER_VERSION
			GathererDetails["author"] = "Norganna"
			GathererDetails["website"] = "http://gathereraddon.com"
			GathererDetails["category"] = MYADDONS_CATEGORY_PROFESSIONS
			GathererDetails["frame"] = "Gatherer"
			GathererDetails["optionsframe"] = "Gatherer.Interface.DialogFrame"

			-- Register the addon in myAddOns
			if(myAddOnsFrame_Register) then
				myAddOnsFrame_Register(GathererDetails, GathererHelp)
			end

			LibSwag.Init()
			LibSwag.RegisterHook(Gatherer.Event.OnSwag)
		end

		if (arg1 and string.lower(arg1) == "gatherer") then
			Gatherer.Config.Load()
			Gatherer.Var.Loaded = true
			Gatherer.Event.OnUpdate(0, true)

			Gatherer.Util.Print("Gatherer v"..Gatherer.Var.Version.." -- Loaded!")

			if (Gatherer.Settings.useMainmap == true) then
				Gatherer_WorldMapDisplay:SetText("Hide Items")
			else
				Gatherer_WorldMapDisplay:SetText("Show Items")
			end

			-- Warning at logon on Gatherer Version number change to check for localization zone change (commented, not needed in 1.12)
			--if ( GetLocale() == "deDE" and (not Gatherer.Settings.Version or (Gatherer.Settings.Version and Gatherer.Settings.Version ~= GATHERER_VERSION ))) then
			--	StaticPopup_Show("GATHERER_VERSION_DIALOG")
			--end

			-- record current version number in order to identify the data for backup utilities
			Gatherer.Settings.Version = GATHERER_VERSION

			-- Get values for World Map once for all
			Gatherer.Var.WmWidth = WorldMapDetailFrame:GetWidth()
			Gatherer.Var.WmHeight = WorldMapDetailFrame:GetHeight()
			Gatherer.Var.WmLevel = WorldMapPlayer:GetFrameLevel()
		end
	elseif ( event == "PLAYER_LOGIN" ) then
		local SETTINGS = Gatherer.Settings
		local useMinimap = (SETTINGS.useMinimap) and "On" or "Off"
		local useMainmap = (SETTINGS.useMainmap) and "On" or "Off"
		local mapMinder = (SETTINGS.mapMinder) and "On" or "Off"
		local minderTime = SETTINGS.minderTime.."s"

		if ( SETTINGS.logInfo and SETTINGS.logInfo == "on" ) then
			Gatherer.Util.Print("[Player: "..Gatherer.Var.PlayerName..", Theme: "..SETTINGS.iconSet..", Mainmap: "..useMainmap..", Minimap: "..useMinimap..", MaxDist: "..SETTINGS.maxDist.." units, NoteCount: "..SETTINGS.number..", Fade: "..SETTINGS.fadePerc.."% at "..SETTINGS.fadeDist.." units, IconDist: "..SETTINGS.miniIconDist.."px on minimap, MapMinder: "..mapMinder.." ("..minderTime.."), Filters: herbs="..Gatherer.Command.GetFilterVal("herbs")..", mining="..Gatherer.Command.GetFilterVal("mining")..", treasure="..Gatherer.Command.GetFilterVal("treasure").."]")
		end

	elseif ( event == "PLAYER_LOGOUT" ) then
		Gatherer.Config.Save()

	elseif ( event == "LEARNED_SPELL_IN_TAB" or event == "SPELLS_CHANGED" ) then
		local numSkills = tonumber(GetNumSkillLines())
		if ( GetNumSkillLines() > 0 ) then
			Gatherer.Util.GetSkills()
		end
	elseif ( event == "SKILL_LINES_CHANGED" ) then
		local numSkills = tonumber(GetNumSkillLines())
		if ( GetNumSkillLines() > 0 ) then
			Gatherer.Util.GetSkills()
		end
	elseif ( event == "CHAT_MSG_ADDON" ) then
		p("Got message:", ...)

	elseif ( event ) then
		Gatherer.Util.ChatPrint("Gatherer Unknown event: "..event)
	end
end

if (not p) then p = function() end end

function Gatherer.Event.OnUpdate(timeDelta, force)
	if (not Gatherer.Var.Loaded) then
		Gatherer.Util.Print("Gatherer not loaded")
		return
	end

	local SETTINGS = Gatherer.Settings

	local needsUpdate = false
	if (not GatherNotes) then
		GatherNotes = { timeDiff=0, checkDiff=0 }
		needsUpdate = true
	else
		GatherNotes.timeDiff = GatherNotes.timeDiff + timeDelta
		if (GatherNotes.timeDiff > Gatherer.Var.NoteUpdateInterval * 10) then
			needsUpdate = true
		end
	end
	if (force) then
		needsUpdate = true
		recalculate = true
	end

	if (needsUpdate) then
		local now = time()
		
		local c, z, px, py = Astrolabe:GetCurrentPlayerPosition()
		if not ( c and z and px and py ) then return end

		local displayNumber = SETTINGS.number
		local maxDist = SETTINGS.maxDist
		maxDist = 1000

		local curPos = 0
		for i, nodeC,nodeZ, nodeID, nodePos, nodeDist, nodeX,nodeY, nodeCount, gtype, nodeHarvested, nodeInspected, nodeSource
		 in Gatherer.Storage.ClosestNodesInfo(c, z, px, py, displayNumber, maxDist, true) do
			
			curPos = curPos + 1
			if (curPos > SETTINGS.number) then break end

			-- need to position and label the corresponding button
			local gatherNote = getglobal("GatherNote"..curPos)
			local gatherNoteTexture = getglobal("GatherNote"..curPos.."Texture")
			Astrolabe:PlaceIconOnMinimap(gatherNote, nodeC, nodeZ, nodeX, nodeY)

			local fadeDist = SETTINGS.fadeDist
			local fadePerc = SETTINGS.fadePerc / 100
			local tracDist = SETTINGS.tracDist or 100
			local opacity = (SETTINGS.alpha or 80) / 100
			
			local iconColor = "normal"

			--TODO Make these proper settings
			local tracDist = 110
			local fadeDist = 500
			local fadePerc = 0.2
			local inspDist = 20
			local inspPerc = 0.2
			local inspTime = 120
			local anonPerc = 0.6
			local opacity = 0.8
			
			local selectedTexture
			local trimTexture = false
			
			if (Gatherer.Icons[nodeID]) then
				selectedTexture = "Interface\\AddOns\\Gatherer\\Icons\\"..Gatherer.Icons[nodeID]
			end

			if (not selectedTexture) then
				local prime = 0
				local pcount = 0
				if (GatherDrops and GatherDrops[nodeID]) then
					for item, count in pairs(GatherDrops[nodeID]) do
	--					if (force) then DEFAULT_CHAT_FRAME:AddMessage("Found "..item.." x "..count) end
						if (item ~= "total" and pcount < count) then
							prime = item
							pcount = count
						end
					end
	--			else
	--				if (force) then DEFAULT_CHAT_FRAME:AddMessage("No data for "..(nodeID or "None")) end
				end

				local primaryName, _, _, _, _, _, _, _, _, nodeTexture = GetItemInfo(prime)
	--			if (force) then DEFAULT_CHAT_FRAME:AddMessage("Best: "..prime.." x "..pcount.." = "..(nodeTexture or "None")) end
				selectedTexture = nodeTexture
				trimTexture = true
			end

			-- Check to see if we found the item
			if (not selectedTexture) then
				selectedTexture = "Interface\\AddOns\\Gatherer\\Shaded\\Red"
				trimTexture = false
			end

			-- If this icon has not been verified
			if (nodeSource) then
				opacity = anonPerc
				iconColor = "red"
			end

			-- If node is within tracking distance
			if (nodeDist < tracDist) then
				if (Gatherer.Util.GetTrackingMode() == Gatherer.Nodes.Objects[nodeID]) then
					selectedTexture = "Interface\\AddOns\\Gatherer\\Shaded\\White"
					trimTexture = false
				end
			end

			-- If we need to fade the icon (because of great distance
			if (fadeDist > 0 and fadePerc > 0 and nodeDist > tracDist) then
				local range = math.max(fadeDist - tracDist, 0)
				local posit = math.min(nodeDist - tracDist, range)
				if (range > 0) then
					local ratio = posit / range
					local fadeLevel = (opacity - fadePerc) * ratio
					opacity = opacity - fadeLevel
				end
			end

			-- If we are within 35 yards of this item, mark it as inspected
			if (nodeDist < inspDist) then
				Gatherer.Storage.SetNodeInspected(nodeC, nodeZ, nodeID, nodePos)
				iconColor = "green"
				opacity = 1

			-- If we've recently seen this node, set its transparency
			elseif (nodeInspected and inspTime > 0 and now - nodeInspected < inspTime) then
				local posit = math.max(now - nodeInspected, 0)
				local ratio = 1 - (posit / inspTime)
				local fadeLevel = (opacity - inspPerc) * ratio
				opacity = opacity - fadeLevel
			end

			-- Set the texture
			gatherNoteTexture:SetTexture(selectedTexture)

			-- Check to see if we need to trim the border off
			if (trimTexture) then
				gatherNoteTexture:SetTexCoord(0.08,0.92,0.08,0.92)
			else
				gatherNoteTexture:SetTexCoord(0,1,0,1)
			end

			-- Put the frame at the right level
			gatherNote:SetFrameLevel(MiniMapTrackingFrame:GetFrameLevel()-1)

			-- If this node is unverified, then make it reddish
			if (iconColor == "red") then
				gatherNoteTexture:SetVertexColor(0.9,0.4,0.4)
			elseif (iconColor == "green") then
				gatherNoteTexture:SetVertexColor(0.4,0.9,0.4)
			elseif (iconColor == "blue") then
				gatherNoteTexture:SetVertexColor(0.4,0.4,0.9)
			else
				gatherNoteTexture:SetVertexColor(1,1,1)
			end
			gatherNoteTexture:SetAlpha(opacity)

--[[
			-- Added to allow hiding if under min distance
			if ( SETTINGS.NoIconOnMinDist ~= nil and SETTINGS.NoIconOnMinDist == 1 ) then
				if ( gDist < (math.floor(sDist)-1) ) then
					gatherNote:Hide()
				else
					gatherNote:Show()
				end
			elseif ( (not SETTINGS.NoIconOnMinDist or SETTINGS.NoIconOnMinDist == 0) and SETTINGS.alphaUnderMinIcon and gDist < (math.floor(sDist)-1) ) then
				if ( SETTINGS.iconSet and SETTINGS.iconSet ~= "iconshade" ) then
					gatherNote:SetAlpha(SETTINGS.alphaUnderMinIcon / 100)
				end
				gatherNote:Show()
			else
				gatherNote:Show()
			end
--]]
		end

		while (curPos < Gatherer.Var.MaxNumNotes) do
			curPos = curPos+1
			Astrolabe:RemoveIconFromMinimap(getglobal("GatherNote"..curPos));
		end
	end
end

function Gatherer.Event.OnSwag(lootType, lootTable, coinAmount, extraData)
	if (lootType ~= "KILL") then
		local gatherType
		if (lootType == "MINE") then gatherType="Mine"
		elseif (lootType == "HERB") then gatherType="Herb"
		elseif (lootType == "SKIN") then gatherType="Skin"
		elseif (lootType == "FISH") then gatherType="Fish"
		elseif (lootType == "OPEN") then gatherType="Treasure"
		end

		local locale = GetLocale()

		local node = "Unknown"
		if (extraData and extraData.tip) then node = extraData.tip end
		local object = Gatherer.Nodes.Names[node]
		if (not object) then 
			DEFAULT_CHAT_FRAME:AddMessage("Node not found: "..node)
			return 
		end

		local objectType = Gatherer.Nodes.Objects[object]
		if (objectType ~= lootType) then
			DEFAULT_CHAT_FRAME:AddMessage("Object Type not the same: "..objectType.."("..object..") != "..lootType)
			return
		end

		Gatherer.Api.AddGather(object, gatherType, storagetip, nil, coinAmount, lootTable)
		DEFAULT_CHAT_FRAME:AddMessage("Recording "..(gatherType or "None").." "..(storagetip or "None").." ["..(object or 0).."]")
	end
end

function Gatherer.Event.ReadBuff(event, fishItem, fishTooltip)
	local record = false
	local nodeName, gatherType, specificType, chatline
	local gatherEventType = 0
	if (not arg1 and not event == "SPELLCAST_STOP") then
		Gatherer.Util.Print("Gatherer: error, no arg recording aborted.")
		Gatherer.Var.RecordFlag = 0
		return
	elseif (arg1) then
		chatline = string.gsub(arg1, GATHERER_NOTEXT, "")
	end

	if ( strfind(event, "CHAT_MSG_LOOT") ) then
	-- process loot received message for fishing node recording
		specificType = Gatherer_FindFishType(fishItem, fishTooltip)
		if ( specificType ) then
			gatherType = "Treasure"
			gatherEventType = 2
			record = true
		end

	elseif( strfind(event, "CHAT_MSG") or strfind(event, "CHAT_MSG_SPELL_SELF_BUFF")) then
		if (string.find(chatline, HERB_GATHER_STRING)) then
			record = true
			nodeName = string.lower(strsub(chatline, HERB_GATHER_LENGTH, HERB_GATHER_END))
			gatherType = "Herb"
			specificType = GatherIcons_TokenConversion[nodeName]

		elseif (string.find(chatline, ORE_GATHER_STRING)) then
			record = true
			nodeName = string.lower(strsub(chatline, ORE_GATHER_LENGTH, ORE_GATHER_END))
			gatherType = "Ore"
			specificType = Gatherer_FindOreType(nodeName)

		elseif (string.find(chatline, TREASURE_GATHER_STRING)) then
			record = true
			nodeName = string.lower(strsub(chatline, TREASURE_GATHER_LENGTH, TREASURE_GATHER_END))
			gatherType = "Treasure"
			local token, canonicalName = Gatherer_FindTreasureType(nodeName)
			specificType = token
			if (not specificType) then return end
			if (canonicalName) then nodeName = Gatherer_GetGatherCanonicalName(gatherIcon) end
		end

	elseif (strfind(event, "UI_ERROR_MESSAGE") or (strfind(event, "SPELLCAST_STOP") and Gatherer.Var.RecordFlag == 1)) then
	-- process event to record, normally not possible gather (low/inexistant skill)
		nodeName = string.lower(Gatherer_ExtractItemFromTooltip())
		if ( Gatherer.Var.CurrentNode and Gatherer.Var.CurrentNode ~= nodeName and event == "SPELLCAST_STOP" and Gatherer.Var.RecordFlag == 1) then
			nodeName = string.lower(Gatherer.Var.CurrentNode)
		end
		if (not arg1 and Gatherer.Var.CurrentAction==nil and event == "SPELLCAST_STOP" and Gatherer.Var.RecordFlag == 1) then
			chatline=""
		elseif (event ~= "UI_ERROR_MESSAGE" ) then
			chatline = Gatherer.Var.CurrentAction
		end
		Gatherer.Util.Debug("nodeName="..nodeName.." chatline="..chatline)

		-- process non gatherable item because of low/lack of skill
		if( nodeName and not nodeName ~= "" and (strfind(chatline, TRADE_HERBALISM) or strfind(chatline, OLD_TRADE_HERBALISM) or strfind(chatline, GATHER_HERBALISM)) ) then -- Herb
			Gatherer.Util.Debug("record Herb")
			record = true
			gatherType = "Herb"
			specificType = GatherIcons_TokenConversion["Herb"][nodeName]
			gatherEventType = 1
		elseif (nodeName and not nodeName ~= "" and strfind(chatline, TRADE_MINING)) then -- Ore
			Gatherer.Util.Debug("record Ore")
			record = true
			gatherType = "Ore"
			specificType = Gatherer_FindOreType(nodeName)
			gatherEventType = 1
		elseif(nodeName and not nodeName ~= "" and (strfind(chatline, TRADE_OPENING) or chatline=="")) then -- Treasure
			Gatherer.Util.Debug("record Treasure")
			record = true
			gatherType = "Treasure"
			local token, canonicalName = Gatherer_FindTreasureType(nodeName)
			specificType = token
--			if (not gatherIcon) then
--				Gatherer.Util.Debug("record Treasuforeachre, no icon abort record")
--				Gatherer.Var.RecordFlag = 0
--				return
--			end
			if ( canonicalName ) then nodeName = canonicalName; end
			gatherEventType = 1
		end
		if (not specificType) then
			Gatherer.Util.Print("Gatherer: no icon identified, aborting record.")
			Gatherer.Var.RecordFlag = 0
			return
		end
		if (event == "SPELLCAST_STOP" and Gatherer.Var.RecordFlag == 1) then
			 gatherEventType = 0
		end
		Gatherer.Var.RecordFlag = 0
	else
		Gatherer.Var.RecordFlag = 0
		return
	end

	if (record == true) then
		Gatherer.Api.AddGather(nodeName, gatherType, specificType, gatherEventType)
		Gatherer.Util.Debug("record type: "..gatherEventType)
	end
end

function Gatherer.Event.OnClick()
	local x, y = GetCursorPosition()
	if ( Minimap.GetEffectiveScale ~= nil ) then
		x = x / Minimap:GetEffectiveScale()
		y = y / Minimap:GetEffectiveScale()
	else
		x = x / Minimap:GetScale()
		y = y / Minimap:GetScale()
	end

	local cx, cy = Minimap:GetCenter()
	x = x + CURSOR_OFFSET_X - cx
	y = y + CURSOR_OFFSET_Y - cy
	if ( sqrt(x * x + y * y) < (Minimap:GetWidth() / 2) ) then
		Minimap:PingLocation(x, y)
	end
end


