﻿
-- module setup
local me = { name = "raidtable"}
local mod = thismod
mod[me.name] = me

--[[
Gui\RaidTable.lua

This class draws the threat table for the raid.
]]

--[[
------------------------------------------------------------------------
						Shared Data
------------------------------------------------------------------------
]]

me.myonupdates = 
{
	updateview = 0.2,
}

me.updateview = function()
	
	if #me.instances > 0 then
		me.instances[1]:rendertable(mod.table.raiddata, mod.table.raiddata, mod.table.raidclasses, nil, "unknown")
	end
	
end 

me.numframes = 0

me.save = 
{
	-- Here are options for all frames
	scale = 1.0,				-- to change the size of just the raid window
	resize = true,				-- whether the table should compress when there are few people in it
	aggrogain = true,			-- show when you will pull aggro from the tank
	aggroloss = true,			-- show when others will pull aggro from you when you are tanking
	columns =
	{
		threat = true,			-- draw "threat" column
		persecond = false,		-- draw "threat per second" column
		percent = true,		-- draw "threat %" column
	},
	[1] = 
	{
		anchor = "topleft",
		colour = 
		{
			r = 0.4,
			g = 0.4,
			b = 1.0,
		},
		position = 
		{
			x = 340,
			y = -340,
		},
		length = 10,
		class = 
		{
			warrior = true, warlock = true, mage = true, druid = true, rogue = true, hunter = true, priest = true, paladin = true, shaman = true,
		},
	},
	[2] = 
	{
		anchor = "topright",
		colour = 
		{
			r = 0.4,
			g = 1.0,
			b = 0.4,
		},
		length = 10,
		position = 
		{
			x = 500,
			y = -340,
		},
		class = 
		{
			warrior = true, warlock = true, mage = true, druid = true, rogue = true, hunter = true, priest = true, paladin = true, shaman = true,
		},
	},
	[3] = 
	{
		anchor = "bottomleft",
		colour = 
		{
			r = 1.0,
			g = 0.4,
			b = 0.4,
		},
		length = 10,
		position = 
		{
			x = 340,
			y = -370,
		},
		class = 
		{
			warrior = true, warlock = true, mage = true, druid = true, rogue = true, hunter = true, priest = true, paladin = true, shaman = true,
		},
	},
	[4] = 
	{
		anchor = "bottomright",
		colour = 
		{
			r = 0.8,
			g = 0.8,
			b = 0.8,
		},
		length = 10,
		position = 
		{
			x = 500,
			y = -370,
		},
		class = 
		{
			warrior = true, warlock = true, mage = true, druid = true, rogue = true, hunter = true, priest = true, paladin = true, shaman = true,
		},
	},
}	

-- Special OnLoad method called by Core.lua
me.onload = function()
	
	---- disabled for the moment
	--if true then return end
	
	local x, item, anchor
		
	for x = 1, me.numframes do
		
		-- create
		item = me.createinstance()
		
		-- apply saved colour scheme
		item:setcolour(me.save[x].colour.r, me.save[x].colour.g, me.save[x].colour.b)
		
		-- apply saved position
		anchor = string.upper(me.save[x].anchor)
		item.gui.header:ClearAllPoints()
		item.gui.header:SetPoint("TOPLEFT", nil, "TOPLEFT", me.save[x].position.x, me.save[x].position.y)
				
	end
	
end

--[[
------------------------------------------------------------------------
					Instance Setup
------------------------------------------------------------------------

The structure is

this = 
{
	gui = [Frame]
	{
		header = [Frame]
		{
			(components of header)
		}
	},
	
	id = (number),
	
	(some instance methods here)
	
	lastupdate = (time),

]]

me.instances = { }  -- numbered list

me.createinstance = function()
	
	local id = 1 + table.getn(me.instances)
	
	local this = 
	{
		["id"] = id,
		lastupdate = 0,
		table = { length = 0, },
				
		-- methods
		creategui = me.creategui,
		setcolour = me.setcolour,
		maximise = me.maximise,
		minimise = me.minimise,
		
		drawtext = me.drawtext,
		inserttolist = me.inserttolist,
		rendertable = me.rendertable,
		reanchortable = me.reanchortable,
		redosubtablelayout = me.redosubtablelayout,
		drawbars = me.drawbars,
	}
	
	table.insert(me.instances, this)
	
	this:creategui()
		
	return this
	
end

--[[
------------------------------------------------------------------------
						Instance Methods
------------------------------------------------------------------------
]]


me.creategui = function(this)

	-- wrapper frame
	this.gui = CreateFrame("Frame", nil, mod.frame)
	this.gui:SetWidth(1)
	this.gui:SetHeight(1)
	this.gui:SetPoint("CENTER", UIParent)
	
	------------------------------------------------------------------------
	-- 				header
	------------------------------------------------------------------------
	
	this.gui.header = mod.gui.createframe(this.gui, 160, 36, true)
	this.gui.header:SetPoint("CENTER", this.gui)
	
	-- identifier string
	this.gui.header.identifier = mod.gui.createframe(this.gui.header, 20, 20, nil)	
	this.gui.header.identifier.text = mod.gui.createfontstring(this.gui.header.identifier, 14, true)
	this.gui.header.identifier.text:SetPoint("CENTER", -1, 0)
	this.gui.header.identifier.text:SetText(this.id)
	this.gui.header.identifier:SetPoint("BOTTOMLEFT", 2, 2)
	
	-- master target string
	this.gui.header.target = mod.gui.createfontstring(this.gui.header, 12, true)
	this.gui.header.target:SetText("Vaelastrasz The Corrupt")
	this.gui.header.target:SetTextColor(1, 1, 0)
	this.gui.header.target:SetPoint("TOPLEFT", 3, -3)
	this.gui.header.target:SetPoint("TOPRIGHT", -3, -3)
	
	-- position string
	this.gui.header.position = mod.gui.createfontstring(this.gui.header, 14, true)
	this.gui.header.position:SetText("15/25")
	this.gui.header.position:SetPoint("LEFT", this.gui.header.identifier, "RIGHT", 0, 0)
	
	-- close button
	this.gui.header.close = mod.gui.createframe(this.gui.header, 20, 20, nil)
	this.gui.header.close.button = mod.gui.createbutton(this.gui.header.close, 12, 12, true)
	this.gui.header.close.button:SetPoint("CENTER", this.gui.header.close)
	this.gui.header.close:SetPoint("BOTTOMRIGHT", this.gui.header, "BOTTOMRIGHT", -3, 2)

	this.gui.header.close.button:SetNormalTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Up")
	this.gui.header.close.button:GetNormalTexture():SetTexCoord(0.25, 0.70, 0.25, 0.75)
	this.gui.header.close.button:SetPushedTexture("Interface\\Buttons\\UI-Panel-MinimizeButton-Down")
	this.gui.header.close.button:GetPushedTexture():SetTexCoord(0.25, 0.70, 0.25, 0.75)
	this.gui.header.close.button:SetHighlightTexture("Interface\\QuestFrame\\UI-QuestTitleHighlight")
	this.gui.header.close.button:GetHighlightTexture():SetBlendMode("ADD")
	this.gui.header.close.button:GetHighlightTexture():SetTexCoord(0, 1.0, -0.1, 1.1)
	
	-- event handlers
	me.createframeeventhandler("OnClick", this.id, this.gui.header.close.button, "close")
	me.createframeeventhandler("OnEnter", this.id, this.gui.header.close.button, "close")
	me.createframeeventhandler("OnLeave", this.id, this.gui.header.close.button, "close")
	
	-- minimise button
	this.gui.header.minimise = mod.gui.createframe(this.gui.header, 20, 20, nil)
	this.gui.header.minimise.button = mod.gui.createbutton(this.gui.header.minimise, 12, 12, true)
	this.gui.header.minimise.button:SetPoint("CENTER", this.gui.header.minimise)
	this.gui.header.minimise:SetPoint("RIGHT", this.gui.header.close, "LEFT", 3, 0)
	
	-- textures
	this.gui.header.minimise.button:SetNormalTexture("Interface\\Buttons\\UI-MinusButton-UP")
	this.gui.header.minimise.button:GetNormalTexture():SetTexCoord(0.2, 0.8, 0.25, 0.8)
	this.gui.header.minimise.button:SetPushedTexture("Interface\\Buttons\\UI-MinusButton-DOWN")
	this.gui.header.minimise.button:GetPushedTexture():SetTexCoord(0.2, 0.8, 0.25, 0.8)
	this.gui.header.minimise.button:SetHighlightTexture("Interface\\QuestFrame\\UI-QuestTitleHighlight")
	this.gui.header.minimise.button:GetHighlightTexture():SetBlendMode("ADD")
	this.gui.header.minimise.button:GetHighlightTexture():SetTexCoord(0, 1.0, -0.1, 1.1)
	
	-- event handlers
	me.createframeeventhandler("OnClick", this.id, this.gui.header.minimise.button, "minimise")
	me.createframeeventhandler("OnEnter", this.id, this.gui.header.minimise.button, "minimise")
	me.createframeeventhandler("OnLeave", this.id, this.gui.header.minimise.button, "minimise")
	
	-- options button
	this.gui.header.options = mod.gui.createframe(this.gui.header, 20, 20, nil)
	this.gui.header.options.button = mod.gui.createbutton(this.gui.header.options, 12, 12, true)
	this.gui.header.options.button:SetPoint("CENTER", this.gui.header.options)
	this.gui.header.options:SetPoint("RIGHT", this.gui.header.minimise, "LEFT", 3, 0)
	
	-- textures
	this.gui.header.options.button:SetNormalTexture("Interface\\ChatFrame\\UI-ChatIcon-Chat-Up")
	this.gui.header.options.button:GetNormalTexture():SetTexCoord(0.2, 0.8, 0.2, 0.8)
	this.gui.header.options.button:SetPushedTexture("Interface\\ChatFrame\\UI-ChatIcon-Chat-Down")
	this.gui.header.options.button:GetPushedTexture():SetTexCoord(0.2, 0.8, 0.2, 0.8)
	this.gui.header.options.button:SetHighlightTexture("Interface\\Buttons\\UI-Common-MouseHilight")
	this.gui.header.options.button:GetHighlightTexture():SetBlendMode("ADD")
	this.gui.header.options.button:GetHighlightTexture():SetTexCoord(0.2, 0.8, 0.2, 0.8)
	
	-- enable dragging of header
	this.gui.header:SetMovable("true")
	this.gui.header:RegisterForDrag("LeftButton")
	this.gui.header:EnableMouse()
	me.createframeeventhandler("OnDragStart", this.id, this.gui.header, "header")
	me.createframeeventhandler("OnDragStop", this.id, this.gui.header, "header")
	
	-- event handlers
	me.createframeeventhandler("OnClick", this.id, this.gui.header.options.button, "options")
	me.createframeeventhandler("OnEnter", this.id, this.gui.header.options.button, "options")
	me.createframeeventhandler("OnLeave", this.id, this.gui.header.options.button, "options")
	
	------------------------------------------------------------------------
	-- 				table
	------------------------------------------------------------------------
	
	this.gui.table = mod.gui.createframe(this.gui, 160, 36, true)
	this:reanchortable()
	
	-- create some columns
 	this.gui.table.name = me.createcolumn("Name", me.columns.name, me.rowspacing, this.gui.table, "LEFT")
	this.gui.table.threat = me.createcolumn("Threat", me.columns.threat, me.rowspacing, this.gui.table, "RIGHT")
	this.gui.table.percent = me.createcolumn("%Max", me.columns.percent, me.rowspacing, this.gui.table, "RIGHT")
	this.gui.table.persecond = me.createcolumn("TPS", me.columns.persecond, me.rowspacing, this.gui.table, "RIGHT")

	-- bars (don't create them yet)
	this.gui.table.bars = { }

	-- redo table width and height
	this:redosubtablelayout()
	this.gui.table:SetHeight(mod.gui.border * 2 + me.rowspacing)
	
end

-- call this if you change the visibility of one of the columns
me.redosubtablelayout = function(this)
	
	local offset = mod.gui.border
	
	-- name column is always shown
	this.gui.table.name.gui:SetPoint("TOPLEFT", this.gui.table, "TOPLEFT", offset, -mod.gui.border)
	offset = offset + me.columns.name
	
	-- threat column
	if me.save.columns.threat == true then
		this.gui.table.threat.gui:Show()
		this.gui.table.threat.gui:SetPoint("TOPLEFT", this.gui.table, "TOPLEFT", offset, -mod.gui.border)
		offset = offset + me.columns.threat
	else
		this.gui.table.threat.gui:Hide()
	end
	
	-- percent column
	if me.save.columns.percent == true then
		this.gui.table.percent.gui:Show()
		this.gui.table.percent.gui:SetPoint("TOPLEFT", this.gui.table, "TOPLEFT", offset, -mod.gui.border)
		offset = offset + me.columns.percent
	else
		this.gui.table.percent.gui:Hide()
	end
	
	-- persecond column
	if me.save.columns.persecond == true then
		this.gui.table.persecond.gui:Show()
		this.gui.table.persecond.gui:SetPoint("TOPLEFT", this.gui.table, "TOPLEFT", offset, -mod.gui.border)
		offset = offset + me.columns.persecond
	else
		this.gui.table.persecond.gui:Hide()
	end
	
	-- width of the combined frame
	this.gui.table:SetWidth(offset + mod.gui.border)
	
	-- redraw bars
	this:drawbars()	
end

-- call this if you change the anchor point of the frame
me.reanchortable = function(this)
	
	this.gui.table:ClearAllPoints()
	
	if me.save[this.id].anchor == "topleft" then
		this.gui.table:SetPoint("TOPLEFT", this.gui.header, "BOTTOMLEFT", 0, mod.gui.border)
	
	elseif me.save[this.id].anchor == "topright" then
		this.gui.table:SetPoint("TOPRIGHT", this.gui.header, "BOTTOMRIGHT", 0, mod.gui.border)
		
	elseif me.save[this.id].anchor == "bottomleft" then
		this.gui.table:SetPoint("BOTTOMLEFT", this.gui.header, "TOPLEFT", 0, - mod.gui.border)
		
	elseif me.save[this.id].anchor == "bottomright" then
		this.gui.table:SetPoint("BOTTOMRIGHT", this.gui.header, "TOPRIGHT", 0, - mod.gui.border)
	end
	
end

--[[
------------------------------------------------------------------------------------------
									TableColumn Class
------------------------------------------------------------------------------------------
]]

me.columns = 
{
	name = 90,
	threat = 45,
	percent = 45,
	persecond = 45,
}
me.rowspacing = 13

me.createcolumn = function(name, width, spacing, parent, justifyh)
	
	local this = 
	{
		["name"] = name,
		["width"] = width,
		["spacing"] = spacing,
		["justifyh"] = justifyh,
		
		entry = { },
		
		-- add methods:
		setcolumnentry = me.setcolumnentry,
	}
	
	this.gui = CreateFrame("Frame", nil, parent)
	this.gui:SetWidth(width)
	this.gui:SetHeight(spacing)
	
	this.gui.heading = mod.gui.createfontstring(this.gui, 12, true)
	this.gui.heading:SetTextColor(1.0, 1.0, 0)
	this.gui.heading:SetText(name)
	this.gui.heading:SetPoint("TOPLEFT", this.gui, "TOPLEFT")
	this.gui.heading:SetWidth(width)
	this.gui.heading:SetJustifyH(justifyh)
	
	return this
end

me.setcolumnentry = function(this, row, value)
	
	-- does entry <row> exist?
	if row > #this.entry then
		
		local newentry
		
		-- create all the missing entries
		for x = #this.entry + 1, row do
			newentry = mod.gui.createfontstring(this.gui, 12, true, "OUTLINE")
			newentry:SetShadowColor(0, 0, 0, 0.3)
			newentry:SetPoint("TOPLEFT", this.gui, "TOPLEFT", 0, 1 -x * this.spacing)
			newentry:SetWidth(this.width)
			newentry:SetJustifyH(this.justifyh)
			
			this.entry[x] = newentry
		end
	end
	
	-- set
	this.entry[row]:SetText(value)
	this.entry[row]:Show()
end

--[[
me.createframeeventhandler(script, tableid, frame, framename)
Extension of the SetScript method for a RaidTable frame. The event will be sent to the <me.frameevent> method.

<script> is the first argument to SetScript, e.g. "OnLeave", "OnClick" etc
<tableid> is the id of the table that the frame belongs to, it should be <this.id> in the calling scope
<frame> is the operand of SetScript, a reference
<framename> is a string which identifies the frame, e.g. "close" for your close button

]]
me.createframeeventhandler = function(script, tableid, frame, framename)
	
	local methodtext = string.format("%s.%s.frameevent(%s, \"%s\", \"%s\")", mod.namespace, me.name, tableid, framename, script)
	
	local method = loadstring(methodtext)
		
	frame:SetScript(script, method)
	
end

--[[
me.frameevent(index, name, script)
This is our handler for frame events.

<index> identifies which RaidTable instance generated the event (namely, <me.instances[index]>)
<name> is the name of the Frame that fired
<script> is the event handler, e.g. "OnClick", "OnEnter"
]]
me.frameevent = function(index, name, script)
	
	if script == "OnEnter" then
		
		-- make a tooltip
		local header, body
		
		header = mod.string.tryget("raidtable", "tooltip", name)
		
		if header == nil then
			header = name
			body = "???"
			
		else
			body = mod.string.get("raidtable", "tooltip", name .. "text")
		end
		
		mod.gui.tooltip(this, header, body)
	
	elseif script == "OnLeave" then
		GameTooltip:Hide()
		
	elseif script == "OnClick" then	
		mod.print(string.format("onclick event fired! name = %s, index = %s.", name, index))
	
	elseif script == "OnDragStart" then
		this:StartMoving()
		
	elseif script == "OnDragStop" then
		this:StopMovingOrSizing()
	
		local _, _, _, x, y = this:GetPoint(1)
		
		me.save[index].position.x = x
		me.save[index].position.y = y
	end
	
end

--[[
myraidtable:setcolour(red, green, blue)
	Changes the colour scheme of the window. We change the colour of all the backdrop borders.
]]
me.setcolour = function(this, red, green, blue)
	
	this.gui.header:SetBackdropBorderColor(red, green, blue)
	
	me.setframeborder(this.gui.header.identifier, red, green, blue)
	me.setframeborder(this.gui.header.close, red, green, blue)
	me.setframeborder(this.gui.header.options, red, green, blue)
	me.setframeborder(this.gui.header.minimise, red, green, blue)
	
	this.gui.table:SetBackdropBorderColor(red, green, blue)
end

me.setframeborder = function(frame, red, green, blue)
	
	frame:SetBackdropBorderColor(red, green, blue)
	frame:SetBackdropColor(red, green, blue)
	
end

--[[
------------------------------------------------------------------------------------------------
									Rendering a Raid Threat Table
------------------------------------------------------------------------------------------------
]]

me.test1 = function()
	
	local threat = 
	{
		bob = 20,
		bil = 70,
		alice = 35,
		rob = 60,
	}
	threat[UnitName("player")] = 50
	
	local tps = 
	{
		bob = 10,
		bil = 15,
		alice = 12,
		rob = 5,
	}
	tps[UnitName("player")] = 14
	
	local class = 
	{
		bob = "warrior",
		bil = "druid",
		alice = "priest",
		rob = "mage",
	}
	class[UnitName("player")] = UnitClass("player")
	
	for x = 1, me.numframes do
		me.instances[x]:rendertable(threat, tps, class, "rob", "range")
	end
	
end

--[[
this only draws the table part.
we don't want it to work out aggrogain itself; someone else should do that work.

this			pointer to the raidtable on which this method is called
threatdata	list of <name> - <threat> pairs
threatperseconddata		list of <name> - <threatpersecond> pairs
classdata	list of <name> - <class> pairs
tank			player with aggro on the mob, or <nil> if it can't be identified.
targetrange	distance to relevant mob. "melee" or "range". This should also be worked out by the mastertarget guy.
]]
me.rendertable = function(this, threatdata, threatperseconddata, classdata, tank, targetrange)
	
	-- step 1: create a sorted list combining threatdata, threatperseconddata, classdata.
	local name, threat, class, threatbaseline
	
	-- 1a) clear our table
	this.table.length = 0
	
	-- 1b) add raid players. They need to have nonzero threat and pass the class filter
	for name, threat in pairs(threatdata) do
		
		-- get class
		local class = classdata[name]
		
		if name == UnitName("player") then
			class = "self"
		
		elseif class == nil and name == UnitName("pet") then
			class = "pet"
		end
		
		if (threat ~= 0) and (class == nil or me.save[this.id].class[class] ~= false) then
			this:inserttolist(name, threat, threatperseconddata[name], class)
		end
	end
		
	-- step 2: add aggro gain / loss, if applicable
	if tank == UnitName("player") then
		
		-- virtual players are melee aggro and range aggro
		if me.save.aggroloss == true then
			
			this:inserttolist(mod.string.get("misc", "meleeaggro"), math.floor(1.1 * threatdata[tank]), math.floor(1.1 * threatperseconddata[tank]), "aggro")
			
			this:inserttolist(mod.string.get("misc", "rangeaggro"), math.floor(1.3 * threatdata[tank]), math.floor(1.3 * threatperseconddata[tank]), "aggro")
		end
		
	elseif tank and me.save.aggrogain == true then
		-- virtual player is aggro gain
		
		if targetrange == "melee" then
			 this:inserttolist(mod.string.get("misc", "aggrogain"), math.floor(1.1 * threatdata[tank]), math.floor(1.1 * threatperseconddata[tank]), "aggro")
			
		else
			this:inserttolist(mod.string.get("misc", "aggrogain"), math.floor(1.3 * threatdata[tank]), math.floor(1.3 * threatperseconddata[tank]), "aggro")
		end
	end
	
	-- step 3: cull list to the maximum size
	this.table.length = math.min(this.table.length, me.save[this.id].length)
	
	-- step 4: add the player at the bottom if he is too low but has nonzero threat
	if threatdata[UnitName("player")] and (threatdata[UnitName("player")] ~= 0) and (threatdata[UnitName("player")] < this.table[this.table.length].threat) then
		this.table[this.table.length].name = UnitName("player")
		this.table[this.table.length].threat = threatdata[UnitName("player")]
		this.table[this.table.length].name = threatperseconddata[UnitName("player")]
		this.table[this.table.length].class = "self"
	end
	
	-- step 5: get the 100% threat value
	if tank then
		threatbaseline = threatdata[tank]
	elseif this.table.length == 0 then
		threatbaseline = 1
	else
		threatbaseline = this.table[1].threat
	end
	
	-- draw!
	this:drawtext(threatbaseline)
	this:drawbars()
	
end

--[[
this:inserttolist(name, threat, threatpersecond)
	Adds the data to the sorted list <this.table>. 
<name>	string; the entitity's name
<threat>	integer; the entity's threat
<threatpersecond> integer; the entity's threat per second.
<class>	string; a player class id, e.g. "warrior", or "self" or "pet" or "aggro"
]]
me.inserttolist = function(this, name, threat, threatpersecond, class)
	
	-- increase length
	this.table.length = this.table.length + 1
	
	-- add new value at the end if there isn't one
	if this.table[this.table.length] == nil then
		this.table[this.table.length] = { }
	end
	
	local restingspot = 1	-- where the new value will end up. default = 1, if it goes up all the way
	
	-- insert (start at the back and work up, pushing lower entries down)
	for x = this.table.length - 1, 1, -1 do
		
		if this.table[x].threat < threat then
			this.table[x + 1].threat = this.table[x].threat
			this.table[x + 1].name = this.table[x].name
			this.table[x + 1].threatpersecond = this.table[x].threatpersecond
			this.table[x + 1].class = this.table[x].class
			
		else
			restingspot = x + 1
			break
		end
	end
	
	-- found its place
	this.table[restingspot].threat = threat
	this.table[restingspot].name = name
	this.table[restingspot].threatpersecond = threatpersecond
	this.table[restingspot].class = class
end

-- threatbaseline is the "100%" value
me.drawtext = function(this, threatbaseline)
	
	-- now try drawing it too!
	for x = 1, this.table.length do
		this.gui.table.name:setcolumnentry(x, this.table[x].name)
		this.gui.table.threat:setcolumnentry(x, me.abbreviateinteger(this.table[x].threat))
		this.gui.table.percent:setcolumnentry(x, me.abbreviateinteger(100 * this.table[x].threat / threatbaseline))
		this.gui.table.persecond:setcolumnentry(x, me.abbreviateinteger(this.table[x].threatpersecond))
	end
	
	-- hide unneeded texts
	local column
	
	for key, _ in pairs(me.columns) do
		column = this.gui.table[key]
		
		for x = this.table.length + 1, #column.entry do
			column.entry[x]:Hide()
		end
	end
	
	-- now cleaning up
	-- 1) recalculate height of table
	-- 2) make lower rows invisible if applicable
	this.gui.table:SetHeight(mod.gui.border * 2 + (this.table.length + 1) * me.rowspacing)
	
end

--[[
me.abbreviateinteger(value)
Abbreviates a large integer, giving at least 3 digit accuracy.
]]
me.abbreviateinteger = function(value)
	
	-- round to integer
	value = math.floor(0.5 + value)
	
	local output = ""
	
	if value < 0 then
		output = "-"
		value = -value
	end
	
	-- 1 to 10k - 1 ("1" to "9999")
	if value < 10000 then
		output = output .. value
		
	-- 10k to 100k - 1 ("10.0k" to "99.9k")
	elseif value < 100000 then
		output = output .. math.floor(value / 100) / 10
		
		if math.fmod(output, 1) == 0 then
			output = output .. ".0"
		end
		
		output = output .. "k"
	
	-- 100k to 10M - 1 ("100k" to "9999k")
	elseif value < 10000000 then
		output = output .. math.floor(value / 1000) .. "k";
		
	-- 10M to 100M - 1 ("10.0M" to "99.9M")
	elseif value < 100000000 then
		output = output .. math.floor(value / 100000) / 10
		
		if math.fmod(output, 1) == 0 then
			output = output .. ".0"
		end
		
		output = output .. "M"
	
	-- 100M +
	else
		output = output .. math.floor(value / 1000000) .. "M";
	end
	
	return output
end

me.drawbars = function(this)
	
	local red, green, blue, maxwidth
	
	local maxthreat = 1
	if this.table.length > 0 then
		maxthreat = this.table[1].threat
	end
	
	for x = 1, this.table.length do
		
		-- 1) create the bar if it doesn't exist
		if this.gui.table.bars[x] == nil then
			this.gui.table.bars[x] = this.gui.table:CreateTexture()
			this.gui.table.bars[x]:SetTexture("Interface\\TargetingFrame\\UI-StatusBar")
			this.gui.table.bars[x]:SetPoint("TOPLEFT", this.gui.table, "TOPLEFT", 1 + mod.gui.border, 1 - mod.gui.border -x * me.rowspacing)
			this.gui.table.bars[x]:SetHeight(me.rowspacing)
		end
		
		-- 2) set the colour
		if RAID_CLASS_COLORS[string.upper(this.table[x].class)] then
			red = RAID_CLASS_COLORS[string.upper(this.table[x].class)].r
			green = RAID_CLASS_COLORS[string.upper(this.table[x].class)].g
			blue = RAID_CLASS_COLORS[string.upper(this.table[x].class)].b
		
		elseif this.table[x].class == "self" or this.table[x].class == "pet" then
			red = 1
			green = 0
			blue = 0
			
		elseif this.table[x].class == "aggro" then
			red = 0
			green = 0
			blue = 1
		
		else
			red = 0.5
			green = 0.5
			blue = 0.5
		end
		
		this.gui.table.bars[x]:SetVertexColor(red, green, blue)
		this.gui.table.bars[x]:Show()
		
		-- 3) set width
		-- for now, just go 100%
		maxwidth = this.gui.table:GetWidth() - 2 * mod.gui.border - 2
		this.gui.table.bars[x]:SetWidth(maxwidth * this.table[x].threat / this.table[1].threat)
		
	end
	
	-- hide unneeded bars
	for x = this.table.length + 1, #this.gui.table.bars do
		this.gui.table.bars[x]:Hide()
	end
	

end
